package ru.yandex.chemodan.queller.celery.control;

import java.util.UUID;

import javax.annotation.PreDestroy;

import org.joda.time.Duration;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.DirectExchange;
import org.springframework.amqp.core.FanoutExchange;
import org.springframework.amqp.core.MessageListener;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.core.Queue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.queller.celery.control.callback.CeleryGenericCallback;
import ru.yandex.chemodan.queller.celery.worker.WorkerId;
import ru.yandex.chemodan.queller.rabbit.NoWorkingRabbitsException;
import ru.yandex.chemodan.queller.rabbit.PoolListener;
import ru.yandex.chemodan.queller.rabbit.RabbitPool;
import ru.yandex.chemodan.queller.rabbit.RabbitQueues;
import ru.yandex.chemodan.queller.rabbit.RoutedMessage;
import ru.yandex.chemodan.queller.support.BenderJsonMessageConverter;
import ru.yandex.commune.json.JsonArray;
import ru.yandex.commune.json.JsonBoolean;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.misc.enums.EnumResolver;
import ru.yandex.misc.ip.Host;

/**
 * @author yashunsky
 */
public class CeleryControl {
    public final static String INPUT_QUEUE_NAME = "control_" + UUID.randomUUID().toString();

    private final String outputExchangeName;
    private final String inputExchangeName;

    private final RabbitPool rabbitPool;

    private final BenderJsonMessageConverter<CeleryApiRequest> converter;

    private MapF<Method, CeleryGenericCallback> callbacks;

    private final PoolListener listener;

    private final Duration controlMessageTtl;

    private final DirectExchange inputExchange;

    private final Queue inputQueue;

    public enum Method {
        CONTROL_ADD_CONSUMER("add_consumer"),
        CONTROL_AUTOSCALE("autoscale"),
        CONTROL_CANCEL_CONSUMER("cancel_consumer"),
        CONTROL_DISABLE_EVENTS("disable_events"),
        CONTROL_ENABLE_EVENTS("enable_events"),
        CONTROL_POOL_GROW("pool_grow"),
        CONTROL_POOL_SHRINK("pool_shrink"),
        CONTROL_SET_POOL("set_pool"),
        CONTROL_RATE_LIMIT("rate_limit"),
        CONTROL_TIME_LIMIT("time_limit"),
        CONTROL_RESET_CONNECTION("reset_connection"),
        INSPECT_ACTIVE("dump_active"),
        INSPECT_ACTIVE_QUEUES("active_queues"),
        INSPECT_CLOCK("clock"),
        INSPECT_CONF("dump_conf"),
        INSPECT_MEMDUMP("memdump"),
        INSPECT_MEMSAMPLE("memsample"),
        INSPECT_PING("ping"),
        INSPECT_REGISTERED("dump_tasks"),
        INSPECT_REPORT("report"),
        INSPECT_RESERVED("dump_reserved"),
        INSPECT_REVOKED("dump_revoked"),
        INSPECT_SCHEDULED("dump_schedule"),
        INSPECT_STATS("stats"),
        INSPECT_STATS_JAVA_WORKER("stats"),
        STATUS("ping");

        public final String methodName;

        Method(final String methodName) {
            this.methodName = methodName;
        }

        public static EnumResolver<Method> R = EnumResolver.er(Method.class);
    }

    public CeleryControl(RabbitPool rabbitPool,
            String outputExchangeName, String inputExchangeName,
            Duration controlMessageTtl,
            Duration workerReplyTtl,
            Duration serviceQueueXExpires)
    {

        this.rabbitPool = rabbitPool;

        this.outputExchangeName = outputExchangeName;
        this.inputExchangeName = inputExchangeName;

        this.controlMessageTtl = controlMessageTtl;

        FanoutExchange outputExchange = new FanoutExchange(outputExchangeName, false, false);
        inputExchange = new DirectExchange(inputExchangeName, false, false);
        inputQueue = RabbitQueues.withExpiration(INPUT_QUEUE_NAME, workerReplyTtl, serviceQueueXExpires);

        rabbitPool.declareExchange(outputExchange);
        rabbitPool.declareExchange(inputExchange);

        declareInputQueueAndBinding();

        converter = new BenderJsonMessageConverter<>(CeleryApiRequest.class);

        callbacks = Cf.concurrentHashMap();

        listener = rabbitPool.createListener();

        listener.setMessageListener((MessageListener) msg -> {
            Option<Ticket> ticket = Ticket.parseSafe(msg.getMessageProperties().getHeaders().get("ticket").toString());

            ticket.forEach(t -> callbacks.getO(Method.R.valueOf(t.method)).forEach(c -> c.getMessage(msg)));
        });
        listener.setQueues(inputQueue);
        listener.start();
    }

    public void declareInputQueueAndBinding() {
        // inputQueue can expire if no workers are connected to mworker
        rabbitPool.declareQueue(inputQueue);
        rabbitPool.declareBinding(BindingBuilder.bind(inputQueue).to(inputExchange).withQueueName());
    }

    @PreDestroy
    public void destroy() {
        listener.stop();
    }

    public void registerCallback(CeleryGenericCallback callback) {
        callbacks.put(callback.getCeleryMethod(), callback);
    }

    public void unregisterCallback(Method method) {
        callbacks.removeO(method);
    }

    private String sendCommand(
            Option<ListF<WorkerId>> destination, Method method, MapF<String, JsonValue> arguments)
    {
        Ticket ticket = Ticket.generate(method.name());

        Option<ListF<String>> serialisedDestination = destination.isPresent()
                ? Option.of(destination.get().map(WorkerId::toString))
                : Option.empty();

        CeleryApiRequest request = new CeleryApiRequest(INPUT_QUEUE_NAME, inputExchangeName,
                method.methodName, arguments, ticket, serialisedDestination);

        MessageProperties messageProperties = new MessageProperties();
        messageProperties.setExpiration(String.valueOf(controlMessageTtl.getMillis()));
        RoutedMessage rm = new RoutedMessage(
                converter.toMessage(request, messageProperties), outputExchangeName, Option.empty());

        try {
            rabbitPool.sendMessages(Cf.list(rm), RabbitPool.MessageType.BROADCAST);
            return ticket.serialize();

        } catch (NoWorkingRabbitsException e) {
            return "";
        }
    }

    public String controlAddConsumer(Option<ListF<WorkerId>> destination, String queueName, String exchange,
            String exchangeType, String routingKey)
    {
        MapF<String, JsonValue> arguments = Cf.hashMap();
        arguments.put("queue", JsonString.valueOf(queueName));
        arguments.put("exchange", JsonString.valueOf(exchange));
        arguments.put("exchange_type", JsonString.valueOf(exchangeType));
        arguments.put("routing_key", JsonString.valueOf(routingKey));
        return sendCommand(destination, Method.CONTROL_ADD_CONSUMER, arguments);
    }

    public String controlAutoScale(Option<ListF<WorkerId>> destination, int max, int min) {
        MapF<String, JsonValue> arguments = Cf.map(
                "max", JsonNumber.valueOf(max),
                "min", JsonNumber.valueOf(min)
        );
        return sendCommand(destination, Method.CONTROL_AUTOSCALE, arguments);
    }

    public String controlCancelConsumer(Option<ListF<WorkerId>> destination, String queueName) {
        return sendCommand(destination, Method.CONTROL_CANCEL_CONSUMER,
                Cf.map("queue", JsonString.valueOf(queueName)));
    }

    public String controlDisableEvents(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.CONTROL_DISABLE_EVENTS, Cf.map());
    }

    public String controlEnableEvents(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.CONTROL_ENABLE_EVENTS, Cf.map());
    }

    public String controlPoolGrow(Option<ListF<WorkerId>> destination, int n) {
        return sendCommand(destination, Method.CONTROL_POOL_GROW, Cf.map("n", JsonNumber.valueOf(n)));
    }

    public String controlPoolShrink(Option<ListF<WorkerId>> destination, int n) {
        return sendCommand(destination, Method.CONTROL_POOL_SHRINK, Cf.map("n", JsonNumber.valueOf(n)));
    }

    public String controlSetPool(Option<ListF<WorkerId>> destination, int pool_size) {
        // available only in customised Celery
        return sendCommand(destination, Method.CONTROL_SET_POOL, Cf.map("pool_size", JsonNumber.valueOf(pool_size)));
    }

    public String controlRateLimit(Option<ListF<WorkerId>> destination, String taskName, String rateLimit) {
        MapF<String, JsonValue> arguments = Cf.map(
                "task_name", JsonString.valueOf(taskName),
                "rate_limit", JsonString.valueOf(rateLimit)
        );
        return sendCommand(destination, Method.CONTROL_RATE_LIMIT, arguments);
    }

    public String controlTimeLimit(Option<ListF<WorkerId>> destination, String taskName, double softSecs,
            double hardSecs)
    {
        MapF<String, JsonValue> arguments = Cf.map(
                "task_name", JsonString.valueOf(taskName),
                "soft", JsonNumber.valueOf(softSecs),
                "hard", JsonNumber.valueOf(hardSecs)
        );
        return sendCommand(destination, Method.CONTROL_TIME_LIMIT, arguments);
    }

    public String controlResetConnection(Option<ListF<WorkerId>> destination, Host newRabbit) {
        return sendCommand(destination, Method.CONTROL_RESET_CONNECTION,
                Cf.map("new_broker", JsonString.valueOf(newRabbit.format())));
    }

    public String inspectActive(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_ACTIVE, Cf.map());
    }

    public String inspectActiveQueues(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_ACTIVE_QUEUES, Cf.map());
    }

    public String inspectClock(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_CLOCK, Cf.map());
    }

    public String inspectConf(Option<ListF<WorkerId>> destination, String withDefault) {
        // set withDefault = "conf" for default conf
        return sendCommand(destination, Method.INSPECT_CONF, Cf.map("with_default", JsonString.valueOf(withDefault)));
    }

    public String inspectMemDump(Option<ListF<WorkerId>> destination, int samples) {
        return sendCommand(destination, Method.INSPECT_MEMDUMP, Cf.map("samples", JsonNumber.valueOf(samples)));
    }

    public String inspectMemSample(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_MEMSAMPLE, Cf.map());
    }

    public String inspectObjGraph(Option<ListF<WorkerId>> destination) {
        //TODO explore Celery method
        return "";
    }

    public String inspectPing(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_PING, Cf.map());
    }

    public String inspectRegistered(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_REGISTERED, Cf.map("taskinfoitems", JsonArray.empty()));
    }

    public String inspectReport(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_REPORT, Cf.map());
    }

    public String inspectReserved(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_RESERVED, Cf.map("safe", JsonBoolean.valueOf(false)));
    }

    public String inspectRevoked(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_REVOKED, Cf.map());
    }

    public String inspectScheduled(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_SCHEDULED, Cf.map("safe", JsonBoolean.valueOf(false)));
    }

    public String inspectStats(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.INSPECT_STATS, Cf.map());
    }

    public String status(Option<ListF<WorkerId>> destination) {
        return sendCommand(destination, Method.STATUS, Cf.map());
    }
}
