package ru.yandex.chemodan.queller.celery.job;

import org.joda.time.DateTime;
import org.joda.time.Seconds;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.json.JsonNull;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @url https://github.com/celery/celery/blob/master/celery/app/amqp.py#L369
 *
 * @author dbrylev
 */
@Bendable
public class CeleryJob extends DefaultObject {
    @BenderPart
    public final TaskId task;
    @BenderPart
    public final String id;

    @BenderPart
    public final int retries;
    @BenderPart
    public final ListF<JsonValue> args;

    private MapF<String, JsonValue> kwargs;
    private Option<CeleryJobContext> context;

    @BenderPart
    public final boolean utc;
    @BenderPart
    public final Option<JsonValue> expires;
    @BenderPart
    public final Option<DateTime> eta;

    @BenderPart
    public final Option<JsonValue> callbacks;
    @BenderPart
    public final Option<JsonValue> errbacks;
    @BenderPart
    public final Option<JsonValue> chord;
    @BenderPart
    public final Option<JsonValue> taskset;

    private Option<Seconds> hardTimeLimit;
    private Option<Seconds> softTimeLimit;

    public CeleryJob(
            TaskId task, String id, int retries,
            ListF<JsonValue> args, MapF<String, JsonValue> kwargs,
            boolean utc, Option<JsonValue> expires, Option<DateTime> eta,
            Option<JsonValue> callbacks, Option<JsonValue> errbacks,
            Option<JsonValue> chord, Option<JsonValue> taskset,
            Option<Seconds> hardTimeLimit, Option<Seconds> softTimeLimit,
            Option<CeleryJobContext> context)
    {
        this.task = task;
        this.id = id;
        this.retries = retries;
        this.args = args;
        this.kwargs = kwargs;
        this.utc = utc;
        this.expires = expires;
        this.eta = eta;
        this.callbacks = callbacks;
        this.errbacks = errbacks;
        this.chord = chord;
        this.taskset = taskset;
        this.hardTimeLimit = hardTimeLimit;
        this.softTimeLimit = softTimeLimit;
        this.context = context;
    }

    public boolean isGlobalQueued() {
        return getContext().exists(ctx -> ctx.globalQueued);
    }

    public CeleryJob withContext(CeleryJobContext context) {
        return new CeleryJob(
                task, id, retries, args, kwargs,
                utc, expires, eta, callbacks, errbacks, chord, taskset,
                hardTimeLimit, softTimeLimit, Option.of(context));
    }

    public CeleryJob withRetriesIncremented() {
        return new CeleryJob(
                task, id, retries + 1, args, kwargs,
                utc, expires, eta, callbacks, errbacks, chord, taskset,
                hardTimeLimit, softTimeLimit, context);
    }

    public String forLog() {
        return task.getId() + "/" + id;
    }

    @BenderPart(name = "timelimit")
    private ListF<JsonValue> serializeTimelimit() {
        Function<Seconds, JsonValue> valueF = duration -> JsonNumber.valueOf(duration.getSeconds());
        return Cf.list(
                hardTimeLimit.<JsonValue>map(valueF).getOrElse(JsonNull.NULL),
                softTimeLimit.<JsonValue>map(valueF).getOrElse(JsonNull.NULL));
    }

    @BenderPart(name = "timelimit")
    private void parseTimelimit(ListF<String> data) {
        this.hardTimeLimit = Cf.Integer.parseSafe(data.get(0)).map(Seconds::seconds);
        this.softTimeLimit = Cf.Integer.parseSafe(data.get(1)).map(Seconds::seconds);
    }

    @BenderPart(name = "kwargs")
    private MapF<String, JsonValue> serializeKwargs() {
        return context.isPresent() ? kwargs.plus1("context", context.get().toJson()) : kwargs;
    }

    @BenderPart(name = "kwargs")
    private void parseKwargs(MapF<String, JsonValue> data) {
        this.kwargs = data.filterKeys(k -> !k.equals("context"));
        this.context = data.getO("context").<JsonObject>cast().map(CeleryJobContext::fromJson);
    }

    public Option<Seconds> getHardTimeLimit() {
        return hardTimeLimit;
    }

    public Option<Seconds> getSoftTimeLimit() {
        return softTimeLimit;
    }

    public MapF<String, JsonValue> getKwargs() {
        return kwargs;
    }

    public Option<CeleryJobContext> getContext() {
        return context;
    }

    public Option<String> getYcrid() {
        return getContext().flatMapO(c -> c.ycrid);
    }

    public Option<String> getHost() {
        return getContext().flatMapO(c -> c.host);
    }
}
