package ru.yandex.chemodan.queller.celery.job;

import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.util.bender.BenderUtils;
import ru.yandex.chemodan.util.json.JsonObjectUtils;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobId;
import ru.yandex.commune.bazinga.impl.JobInfoValue;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.worker.BazingaHostPort;
import ru.yandex.commune.bazinga.pg.storage.JobRunInfo;
import ru.yandex.commune.bazinga.pg.storage.shard.JobsPartitionShardResolver;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.json.JsonArray;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;

/**
 * @author dbrylev
 */
public class CeleryOnetimeJobConverter {

    public static OnetimeJob convertFromCelery(
            CeleryTask task, CeleryJob job, JobStatus status, Option<Instant> scheduleTime,
            JobsPartitionShardResolver partitioning)
    {
        return convertFromCelery(task, job, status, Option.empty(), Option.empty(), Option.empty(),
                scheduleTime, partitioning);
    }

    public static OnetimeJob convertFromCelery(
            CeleryTask task, CeleryJob job, JobRunInfo runInfo, Option<Instant> scheduleTime,
            JobsPartitionShardResolver partitioning)
    {
        return convertFromCelery(task, job, runInfo.status, runInfo.started, runInfo.finished, runInfo.workerId,
                scheduleTime, partitioning);
    }

    public static OnetimeJob convertFromCelery(
            CeleryTask task, CeleryJob job, JobStatus status, Option<Instant> start, Option<Instant> finish,
            Option<BazingaHostPort> workerId, Option<Instant> scheduleTime,
            JobsPartitionShardResolver partitioning)
    {
        CeleryJobContext context = job.getContext().getOrElse(CeleryJobContext.empty());

        boolean saveError = status == JobStatus.READY || status == JobStatus.FAILED;

        JobInfoValue info = new JobInfoValue(
                status, start, finish, workerId,
                saveError ? context.error : Option.empty(),
                saveError ? context.traceback : Option.empty(),
                Option.empty(), Option.empty());

        Option<Instant> scheduled = context.scheduled;
        Option<String> activeUid = getActiveUid(
                job, status, task.activeUidBehavior.getDropType());

        return new OnetimeJob(
                getJobId(job),
                Option.of(context.created.getOrElse(Instant.now())),
                scheduleTime.orElse(scheduled).getOrElse(Instant.now()),
                context.activeUid.map(partitioning::partitionByActiveUid),
                activeUid, getParametersWithYcrid(job), info, getAttempt(job),
                Cf.set(), Option.empty(), task.priority, Option.empty());
    }

    public static CeleryJob convertToCelery(OnetimeJob job, boolean globalQueued) {
        JsonObject parameters = JsonObject.parseObject(job.getParameters());

        CeleryJobContext context = new CeleryJobContext(
                globalQueued, Option.of(job.getCreationTime()), Option.of(job.getScheduleTime()),
                job.getActiveUniqueIdentifier(),
                parameters.getO("ycrid").map(JsonObjectUtils::getValueAsText),
                parameters.getO("host").map(JsonObjectUtils::getValueAsText),
                Option.empty(),
                job.getValue().getStart(),
                job.getValue().getFinish(),
                job.getValue().getExceptionMessage(),
                job.getValue().getStackTrace(), Option.empty());

        return new CeleryJob(
                job.getTaskId(), job.getJobId().toSerializedString(),
                getRetires(job),
                Cf.x(((JsonArray) parameters.get("args")).getArray()),
                Cf.toMap(((JsonObject) parameters.get("kwargs")).getObject()),
                true, Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.of(context));
    }

    public static FullJobId getJobId(CeleryJob job) {
        return new FullJobId(job.task, new JobId(UUID.fromString(job.id)));
    }

    public static String getParameters(CeleryJob job) {
        return BenderUtils.serialize(new JsonObject(getParametersRaw(job)));
    }

    private static String getParametersWithYcrid(CeleryJob job) {
        Tuple2List<String, JsonValue> fields = getParametersRaw(job)
                .plus(job.getYcrid().map(rid -> Tuple2.tuple("ycrid", JsonString.valueOf(rid))))
                .plus(job.getHost().map(host -> Tuple2.tuple("host", JsonString.valueOf(host))));

        return BenderUtils.serialize(new JsonObject(fields));
    }

    private static Tuple2List<String, JsonValue> getParametersRaw(CeleryJob job) {
        Tuple2List<String, JsonValue> fields = Tuple2List.arrayList();

        fields.add("args", new JsonArray(job.args));
        fields.add("kwargs", new JsonObject(job.getKwargs()));

        return fields;
    }

    public static Option<Integer> getAttempt(CeleryJob job) {
        return Option.when(job.retries > 0, job.retries - 1);
    }

    private static int getRetires(OnetimeJob job) {
        return job.getAttempt().getOrElse(-1) + 1;
    }

    private static Option<String> getActiveUid(
            CeleryJob job, JobStatus status, ActiveUidDropType dropType)
    {
        Option<String> activeUid = job.getContext().flatMapO(ctx -> ctx.activeUid);

        if (!activeUid.isPresent()) {
            return Option.empty();
        }
        switch (dropType) {
            case WHEN_RUNNING:
                return status == JobStatus.READY ? activeUid : Option.empty();

            case WHEN_FINISHED:
                return status.isCompleted() ? Option.empty() : activeUid;

            default:
                throw new IllegalStateException("Unknown drop type: " + dropType);
        }
    }
}
