package ru.yandex.chemodan.queller.rabbit;

import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.TimeoutException;

import com.rabbitmq.client.AlreadyClosedException;
import com.rabbitmq.client.Consumer;
import org.springframework.amqp.core.AcknowledgeMode;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.rabbit.connection.CachingConnectionFactory;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.queller.rabbit.patchedSpringAmqp.BatchMessageListener;
import ru.yandex.chemodan.queller.rabbit.patchedSpringAmqp.BatchMessageListenerContainer;
import ru.yandex.chemodan.queller.rabbit.patchedSpringAmqp.BlockingQueueConsumer;
import ru.yandex.misc.ip.Host;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
public class ConnectedListener {
    private static final Logger logger = LoggerFactory.getLogger(ConnectedListener.class);

    private final RabbitConnection connection;
    private final RabbitConnectionPojo connectionData;
    private final BatchMessageListenerContainer listener;
    private final CachingConnectionFactory factory;

    private final Object channelAcquisitionMonitor = new Object();

    public ConnectedListener(RabbitConnection connection) {
        this.connection = connection;
        this.connectionData = connection.connectionData;

        connection.addListener(this);

        factory = new CachingConnectionFactory(connectionData.host.toString(), connectionData.port.getPort());

        factory.setUsername(connectionData.username);
        factory.setPassword(connectionData.password);
        factory.setVirtualHost(connectionData.virtualHost);

        try {
            factory.afterPropertiesSet();
        } catch (Exception e) {
            logger.error("Connection factory for {} configuration failed: {}", connectionData.host, e);
        }

        this.listener = new BatchMessageListenerContainer(factory, channelAcquisitionMonitor);

        Assert.lt(this.listener.getShutdownTimeout(), this.connectionData.listenerStopTimeout.getMillis(),
                "If shutdownTimeout > listenerStopTimeout, the listener will always stop by timeout exception");
    }

    public Host getHost() {
        return connectionData.host;
    }

    public final void setExclusive(boolean exclusive) {
        listener.setExclusive(exclusive);
    }

    public void setQueues(Queue... queues) {
        listener.setQueues(queues);
    }

    public void setConcurrentConsumers(final int concurrentConsumers) {
        listener.setConcurrentConsumers(concurrentConsumers);
    }

    public void setMaxConcurrentConsumers(int maxConcurrentConsumers) {
        listener.setMaxConcurrentConsumers(maxConcurrentConsumers);
    }

    public void setPrefetchCount(int prefetchCount) {
        listener.setPrefetchCount(prefetchCount);
    }

    public void setTxSize(int txSize) {
        listener.setTxSize(txSize);
    }

    public void setReceiveTimeout(long receiveTimeoutMs) {
        listener.setReceiveTimeout(receiveTimeoutMs);
    }

    public void setMessageListener(Object messageListener) {
        if (messageListener instanceof BatchMessageListener) {
            listener.setBatchMessageListener((BatchMessageListener) messageListener);
        } else {
            listener.setMessageListener(messageListener);
        }

    }

    public void setAcknowledgeMode(AcknowledgeMode acknowledgeMode) {
        listener.setAcknowledgeMode(acknowledgeMode);
    }

    public synchronized void start() {
        innerStart();
    }

    private void innerStart() {
        if (!connection.runWithTimeout(listener::start, connectionData.listenerStartTimeout)) {
            logger.error("Failed to start listener for {} @ {} because of timeout",
                    Arrays.asList(listener.getQueueNames()), connectionData.host);
        }
    }

    public synchronized void stop() {
        innerStop();
    }

    private void innerStop() {
        if (!connection.runWithTimeout(listener::stop, connectionData.listenerStopTimeout)) {
            logger.error("Failed to stop listener for {} @ {} because of timeout",
                    Arrays.asList(listener.getQueueNames()), connectionData.host);
        }
    }

    public synchronized void restart() {
        innerStop();
        innerStart();
    }

    public void startIfNotRunning() {
        if (listener.isRunning()) {
            innerStart();
        }
    }

    public void closeUnexpectedChannels() {
        /*
            springworks.amqp can silently swallow IoException on consumer's cancel,
            so we use RabbitHackUtils to get the list of real consumers and channels.
            If a the channel has an unexpected consumer, or has no consumers, it is closed.
            NB: in our system we have 1 or 0 consumers per channel
         */

        RabbitHackUtils.getChannels.apply(factory).forEach(channelN -> {
            synchronized (channelAcquisitionMonitor) {
                SetF<Consumer> expectedConsumers =
                        listener.getConsumers().keys().map(BlockingQueueConsumer::getConsumer).unique();
                ListF<Consumer> realConsumers = RabbitHackUtils.getConsumers.apply(channelN);

                // TODO: channels without consumers can lock tasks, but not every channel without consumer can be closed
                if (/*realConsumers.isEmpty() || */realConsumers.exists(expectedConsumers.containsF().notF())) {
                    connection.runWithTimeout(() -> {
                        try {
                            channelN.close();
                        } catch (AlreadyClosedException ignore) {
                        } catch (IOException | TimeoutException e) {
                            logger.error("Failed closing unexpected channel {}", e);
                        }
                    }, connectionData.channelCloseTimeout);
                }
            }
        });
    }
}
