package ru.yandex.chemodan.queller.rabbit;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.joda.time.Duration;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.Exchange;
import org.springframework.amqp.core.Queue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.chemodan.queller.celery.monitoring.CeleryMetrics;
import ru.yandex.chemodan.queller.celery.worker.WorkerStateProviderHolder;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.ip.Host;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author yashunsky
 */
public class RabbitPool extends DelayingWorkerServiceBeanSupport {

    public static final Comparator<RabbitConnectionHealth> RABBIT_CONNECTION_HEALTH_COMPARATOR =
            (Comparator<RabbitConnectionHealth>) Comparator.<RabbitConnectionHealth>constEqualComparator()
                    .thenComparing(h -> h.isActive)
                    .thenComparing(h -> h.isConnected)
                    .thenComparing(h -> h.readyToReceive)
                    .thenComparing(h -> h.sendingErrorRate.isOk)
                    .thenComparing(h -> h.sendingDuration.isOk)
                    .thenComparing(h -> h.pingTimeoutPerMinute.isOk);

    private static final Logger logger = LoggerFactory.getLogger(RabbitPool.class);

    private final MapF<Host, RabbitConnection> rabbitConnectionsByHost;
    private final int maxUnusedRabbitsCount;

    private ListF<PoolListener> maintainedListeners;

    private final CeleryMetrics celeryMetrics;
    private final ExecutorService sendTasksExecutor;
    private final ExecutorService maintenanceExecutor;

    public enum MessageType {
        BROADCAST,
        CONFIRMED,
        UNCONFIRMED
    }

    public RabbitPool(ListF<Host> hosts, IpPort port,
            String username, String password, String virtualHost,
            Duration declarationTimeout,
            Duration getPropertiesTimeout,
            Duration batchSendingTimeout,
            Duration batchConfirmationTimeout,
            Duration threadSleepInLoop,
            Duration listenerStartTimeout,
            Duration listenerStopTimeout,
            Duration channelAbortTimeout,
            Duration poolMaintenancePeriod,
            Duration connectionMaintenancePeriod,
            Duration serviceQueuesXExpires,
            int executorPoolSize,
            int maxUnusedRabbitsCount,
            WorkerStateProviderHolder celeryMonitor,
            CeleryMetrics celeryMetrics)
    {
        this.celeryMetrics = celeryMetrics;

        this.rabbitConnectionsByHost = hosts
                .map(host -> new RabbitConnectionPojo(host, port, virtualHost, username, password,
                        declarationTimeout, getPropertiesTimeout, batchSendingTimeout,
                        batchConfirmationTimeout, threadSleepInLoop,
                        listenerStartTimeout, listenerStopTimeout, channelAbortTimeout, executorPoolSize))
                .map(connectionData -> new RabbitConnection(
                        connectionData, celeryMonitor, this.celeryMetrics,
                        connectionMaintenancePeriod, serviceQueuesXExpires))
                .toMap(rc -> rc.connectionData.host, rc -> rc);
        this.maxUnusedRabbitsCount = maxUnusedRabbitsCount;

        this.maintainedListeners = Cf.arrayList();

        this.sendTasksExecutor = Executors.newCachedThreadPool();
        this.maintenanceExecutor = Executors.newCachedThreadPool();

        setSleepBeforeFirstRun(false);
        setDelay(poolMaintenancePeriod);
    }

    public synchronized PoolListener createListener() {
        PoolListener permanentListener = new PoolListener(rabbitConnectionsByHost.values().toList());
        maintainedListeners.add(permanentListener);

        return permanentListener;
    }

    public void declareExchange(Exchange exchange) {
        rabbitConnectionsByHost.values().forEach(r -> r.declareExchange(exchange));
    }

    public void declareQueue(Queue queue) {
        rabbitConnectionsByHost.values().forEach(r -> r.declareQueue(queue));
    }

    public void declareBinding(Binding binding) {
        rabbitConnectionsByHost.values().forEach(r -> r.declareBinding(binding));
    }

    public void deleteQueue(String queueName) {
        rabbitConnectionsByHost.values().forEach(r -> r.deleteQueue(queueName));
    }

    public QueueState aggregateQueueStates(String queueName) {
        return getWorkingRabbits().foldLeft(new QueueState(new Queue(queueName), 0, 0), (agg, c) -> {
            QueueState state = c.getQueueState(queueName);

            return new QueueState(
                    state.queue,
                    state.messageCount + agg.messageCount,
                    state.consumerCount + agg.consumerCount);
        });
    }

    public MapF<Host, RabbitConnectionHealth> getHealth() {
        return rabbitConnectionsByHost.mapValues(RabbitConnection::getHealth);
    }

    @Override
    public void start() {
        super.start();
        rabbitConnectionsByHost.forEach((host, rc) -> rc.start());
    }

    @Override
    public void stop() {
        sendTasksExecutor.shutdownNow();
        super.stop();
    }

    @Override
    protected void execute() throws Exception {
        SetF<Host> hostsToUse = getHealth().values()
                .sorted(RABBIT_CONNECTION_HEALTH_COMPARATOR).drop(maxUnusedRabbitsCount)
                .map(rch -> Host.parse(rch.host)).unique();

        rabbitConnectionsByHost.forEach((host, connection) -> {
            if (!connection.isActive()) {
                logger.error("Connection @ {} inactive because of timeout", host);
            }
            connection.setForceUse(hostsToUse.containsTs(host));
        });

        maintainedListeners.forEach(l -> maintenanceExecutor.submit(l::startIfNotRunning));
    }


    public void sendMessageConfirmed(RoutedMessage message) {
        sendMessage(message, true);
    }

    public void sendMessage(RoutedMessage message, boolean confirmed) {
        if (confirmed) {
            if (!sendMessages(Cf.list(message), MessageType.CONFIRMED).equals(Cf.list(true))) {
                throw new RuntimeException("Message sending was not confirmed");
            }
        } else {
            sendMessages(Cf.list(message), MessageType.UNCONFIRMED);
        }
    }

    public ListF<Boolean> sendMessages(ListF<RoutedMessage> routedMessages, MessageType messageType) {
        ListF<RabbitConnection> workingRabbits = getWorkingRabbitsOrThrow();

        Function1V<ListF<SendResult>> throwErrF = rs -> rs.filterMap(SendResult::getError).forEach(e -> { throw e; });

        if (messageType == MessageType.CONFIRMED || messageType == MessageType.UNCONFIRMED) {
            RabbitConnection rc = workingRabbits.get(Random2.R.nextInt(workingRabbits.size()));
            ListF<SendResult> res = rc.sendMessages(routedMessages, messageType == MessageType.CONFIRMED);

            throwErrF.apply(res);

            return res.map(r -> r.status == SendResult.Status.SENT_CONFIRMED);
        }

        if (messageType == MessageType.BROADCAST) {
            workingRabbits.forEach(rc -> throwErrF.apply(rc.sendMessages(routedMessages, false)));
        }

        return routedMessages.map(rm -> false);
    }

    public ListF<SendResult> sendToMostFreeRabbitsWithConfirmation(ListF<RoutedMessage> messages) {
        return sendToRankedRabbitsWithConfirmation(messages, qs -> Option.of(qs.messageCount));
    }

    public ListF<SendResult> sendToMostConsumedRabbitsWithConfirmation(ListF<RoutedMessage> messages) {
        return sendToRankedRabbitsWithConfirmation(messages,
                qs -> Option.when(qs.consumerCount > 0, () -> qs.messageCount / qs.consumerCount));
    }

    private ListF<SendResult> sendToRankedRabbitsWithConfirmation(
            ListF<RoutedMessage> messages, Function<QueueState, Option<Integer>> ranker)
    {
        if (messages.isEmpty()) return Cf.list();

        ListF<RabbitConnection> rabbits = Random2.R.shuffle(getWorkingRabbitsOrThrow());

        ListF<String> queues = messages.map(m -> m.queueName.getOrThrow("Not supported")).stableUnique();

        MapF<String, Host> mostConsumedHostByQueue = queues.iterator()
                .flatMap(q -> rabbits.zipWith(c -> c.getQueueState(q)).iterator())
                .filterMap(qs -> ranker.apply(qs.get2()).map(rank -> Tuple3.tuple(qs.get1(), qs.get2(), rank)))
                .toList()
                .sortedBy(Tuple3::get3)
                .stableUniqueBy(qs -> qs.get2().queue.getName())
                .toMap(qs -> qs.get2().queue.getName(), qs -> qs.get1().host);

        MapF<Host, RabbitConnection> rabbitByHost = rabbits.toMapMappingToKey(c -> c.host);

        MapF<Host, ListF<RoutedMessageWithIndex>> hostMessages = messages
                .zipWithIndex().map(RoutedMessageWithIndex::new)
                .zipWithFlatMapO(m -> mostConsumedHostByQueue.getO(m.message.queueName.get()))
                .groupBy2();

        SendResult[] result = Cf.repeat(SendResult.noConsumers(), messages.size()).toArray(SendResult.class);

        Function2V<Host, ListF<RoutedMessageWithIndex>> sendF = (host, msgs) -> {
            IteratorF<SendResult> res = rabbitByHost.getOrThrow(host)
                    .sendMessages(msgs.map(r -> r.message), true).iterator();

            msgs.forEach(msg -> result[msg.index] = res.next());
        };

        ListF<Host> hosts = hostMessages.keys();

        CompletableFuture<ListF<Void>> others = CompletableFutures.allOf(
                hosts.drop(1).map(host -> CompletableFuture.runAsync(
                        () -> sendF.apply(host, hostMessages.getTs(host)), sendTasksExecutor)));

        hosts.firstO().forEach(host -> sendF.apply(host, hostMessages.getTs(host)));

        CompletableFutures.join(others);

        return Cf.x(result);
    }

    public ListF<RabbitConnection> getWorkingRabbits() {
        return rabbitConnectionsByHost.values().toList().filter(RabbitConnection::canBeUsed);
    }

    public ListF<RabbitConnection> getWorkingRabbitsOrThrow() {
        ListF<RabbitConnection> cs = getWorkingRabbits();

        if (cs.isEmpty()) {
            logger.error("No working rabbits found");
            throw new NoWorkingRabbitsException();
        }
        return cs;
    }
}
