package ru.yandex.chemodan.queller.rabbit;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.queller.celery.monitoring.CeleryMetrics;
import ru.yandex.chemodan.queller.celery.monitoring.CeleryMetricsContextConfiguration;
import ru.yandex.chemodan.queller.celery.worker.WorkerStateProviderHolder;
import ru.yandex.chemodan.queller.utils.ConductorUtils;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.commune.dynproperties.DynamicPropertiesContextConfiguration;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.inside.admin.conductor.ConductorContextConfiguration;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.ip.Host;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
@Configuration
@Import({
        ConductorContextConfiguration.class,
        CeleryMetricsContextConfiguration.class,
        DynamicPropertiesContextConfiguration.class,
})
public class RabbitPoolContextConfiguration {

    @Autowired
    private DynamicPropertyManager dynamicPropertyManager;

    @Bean
    public WorkerStateProviderHolder celeryMonitorHolder() {
        return new WorkerStateProviderHolder();
    }

    @Bean
    public RabbitPool rabbitPool(
            LocationResolver locationResolver,
            @Value("${queller.rabbit.hosts}") String groupsOrHosts,
            @Value("${queller.rabbit.hosts.myDcOnly}") boolean myDcOnly,
            @Value("${queller.rabbit.port}") IpPort port,
            @Value("${queller.rabbit.username}") String username,
            @Value("${queller.rabbit.password}") String password,
            @Value("${queller.rabbit.password.file}") String passwordFile,
            @Value("${queller.rabbit.virtualHost}") String virtualHost,
            @Value("${queller.rabbit.maxUnusedCount}") int maxUnusedRabbitsCount,
            @Value("${queller.rabbit.connection.declarationTimeout}") Duration declarationTimeout,
            @Value("${queller.rabbit.connection.getPropertiesTimeout}") Duration getPropertiesTimeout,
            @Value("${queller.rabbit.connection.batchSendingTimeout}") Duration batchSendingTimeout,
            @Value("${queller.rabbit.connection.batchConfirmationTimeout}") Duration batchConfirmationTimeout,
            @Value("${queller.rabbit.connection.threadSleepInLoop}") Duration threadSleepInLoop,
            @Value("${queller.rabbit.connection.listenerStartTimeout}") Duration listenerStartTimeout,
            @Value("${queller.rabbit.connection.listenerStopTimeout}") Duration listenerStopTimeout,
            @Value("${queller.rabbit.connection.channelCloseTimeout}") Duration channelCloseTimeout,
            @Value("${queller.rabbit.poolMaintenancePeriod}") Duration poolMaintenancePeriod,
            @Value("${queller.rabbit.connectionMaintenancePeriod}") Duration connectionMaintenancePeriod,
            @Value("${queller.rabbit.serviceQueuesXExpires}") Duration serviceQueuesXExpires,
            @Value("${queller.rabbit.connection.executorPoolSize}") int executorPoolSize,
            WorkerStateProviderHolder celeryMonitor, CeleryMetrics celeryMetrics)
    {
        dynamicPropertyManager.addStaticFields(RabbitConnection.class);

        Assert.isTrue(RabbitHackUtils.isLoaded());

        return new RabbitPool(
                ConductorUtils.resolveHostsFromString(locationResolver, groupsOrHosts, myDcOnly ?
                        locationResolver.resolveLocation().dcName : Option.empty())
                        .map(Host::parse), port,
                username, resolvePassword(username, password, passwordFile), virtualHost,
                declarationTimeout, getPropertiesTimeout, batchSendingTimeout,
                batchConfirmationTimeout, threadSleepInLoop,
                listenerStartTimeout, listenerStopTimeout, channelCloseTimeout,
                poolMaintenancePeriod, connectionMaintenancePeriod, serviceQueuesXExpires, executorPoolSize,
                maxUnusedRabbitsCount, celeryMonitor, celeryMetrics);
    }

    private static String resolvePassword(String username, String password, String passwordFile) {
        if (password.isEmpty()) {
            ListF<String> passwords = new File2(passwordFile).asReaderTool().readLines().filterMap(line -> {
                String[] split = line.split(":");

                return Option.when(split.length == 2 && split[0].equals(username), () -> split[1]);
            });
            return passwords.lastO().getOrThrow("No password for user " + username + " found in " + passwordFile);
        }
        return password;
    }
}
