package ru.yandex.chemodan.queller.worker;

import org.joda.time.Duration;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.FanoutExchange;
import org.springframework.amqp.core.MessageListener;
import org.springframework.amqp.core.Queue;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.DependsOn;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.queller.celery.control.CeleryApiRequest;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.celery.worker.WorkerId;
import ru.yandex.chemodan.queller.celery.worker.WorkerName;
import ru.yandex.chemodan.queller.rabbit.PoolListener;
import ru.yandex.chemodan.queller.rabbit.RabbitPool;
import ru.yandex.chemodan.queller.rabbit.RabbitPoolContextConfiguration;
import ru.yandex.chemodan.queller.rabbit.RabbitQueues;
import ru.yandex.chemodan.queller.support.BenderJsonMessageConverter;
import ru.yandex.commune.bazinga.impl.TaskOverridesManager;
import ru.yandex.commune.bazinga.impl.worker.WorkerTaskRegistry;
import ru.yandex.commune.bazinga.impl.worker.WorkerTaskRunner;
import ru.yandex.misc.monica.annotation.MonicaStaticRegistry;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.spring.Service;

/**
 * @author dbrylev
 */
@Configuration
@Import({
        BazingaBeansContextConfiguration.class,
        RabbitPoolContextConfiguration.class,
})
public class CeleryJavaWorkerContextConfiguration {

    @Bean
    public CeleryJavaWorkerConfiguration celeryJavaWorkerConfiguration(
            @Value("${celery.worker.name}") String name,
            @Value("${celery.worker.queue}") String queue)
    {
        WorkerId workerId = new WorkerId(new WorkerName(name), Option.of(HostnameUtils.localhost()));

        return new CeleryJavaWorkerConfiguration(workerId, Cf.list(queue.split(",")));
    }

    @Bean
    public CeleryJavaWorker celeryJavaWorker(
            CeleryJavaWorkerConfiguration configuration,
            WorkerTaskRunner taskRunner,
            WorkerTaskRegistry taskRegistry,
            TaskOverridesManager taskOverridesManager,
            RabbitPool rabbitPool)
    {
        CeleryJavaWorkerMetrics metrics = new CeleryJavaWorkerMetrics();

        MonicaStaticRegistry.register(metrics, "celeryJavaWorker");

        return new CeleryJavaWorker(
                configuration, taskRunner,
                taskRegistry, taskOverridesManager,
                rabbitPool, new BenderJsonMessageConverter<>(CeleryJob.class), metrics);
    }

    @Bean
    public PoolListener celeryJavaTaskListenerContainer(
            RabbitPool rabbitPool,
            CeleryJavaWorker celeryJavaWorker,
            @Value("${celery.worker.prefetchCount}") int prefetchCount,
            @Value("${celery.worker.txSize}") int txSize,
            @Value("${celery.worker.consumers}") int consumers,
            @Value("${celery.worker.maxConsumers}") int maxConsumers)
    {
        ListF<Queue> queues = celeryJavaWorker.configuration.queues.map(RabbitQueues::durable);

        PoolListener container = rabbitPool.createListener();
        container.setExclusive(false);
        container.setQueues(queues.toArray(Queue.class));

        container.setConcurrentConsumers(consumers);
        container.setMaxConcurrentConsumers(maxConsumers);
        container.setPrefetchCount(prefetchCount);
        container.setTxSize(txSize);

        BenderJsonMessageConverter<CeleryJob> converter = new BenderJsonMessageConverter<>(CeleryJob.class);

        container.setMessageListener((MessageListener) msg -> celeryJavaWorker.handleJob(converter.fromMessage(msg)));

        queues.forEach(rabbitPool::declareQueue);

        return container;
    }

    @Bean
    public PoolListener celeryJavaInspectListener(
            RabbitPool rabbitPool,
            CeleryJavaWorker celeryJavaWorker,
            @Value("${queller.celery.outputExchangeName}") String inputExchangeName,
            @Value("${queller.celery.workerReplyTtl}") Duration workerReplyTtl,
            @Value("${queller.celery.serviceQueuesXExpires}") Duration serviceQueuesXExpires)
    {
        Queue queue = RabbitQueues.withExpiration(
                celeryJavaWorker.configuration.workerId.serialize() + ".celery.pidbox",
                workerReplyTtl, serviceQueuesXExpires);

        FanoutExchange exchange = new FanoutExchange(inputExchangeName, false, false);

        PoolListener container = rabbitPool.createListener();
        container.setQueues(queue);

        BenderJsonMessageConverter<CeleryApiRequest> converter = new BenderJsonMessageConverter<>(CeleryApiRequest.class);

        container.setMessageListener((MessageListener) msg -> celeryJavaWorker.inspectStats(converter.fromMessage(msg)));

        rabbitPool.deleteQueue(queue.getName());
        rabbitPool.declareQueue(queue);
        rabbitPool.declareExchange(exchange);
        rabbitPool.declareBinding(BindingBuilder.bind(queue).to(exchange));

        return container;
    }

    @Bean
    @DependsOn("bazingaWorkerApp")
    public Service celeryJavaTaskListenerContainerRunner(
            @Qualifier("celeryJavaTaskListenerContainer") PoolListener tasksListener,
            @Qualifier("celeryJavaInspectListener") PoolListener inspectListener)
    {
        return new Service() {
            public void start() { Cf.list(inspectListener, tasksListener).forEach(PoolListener::start); }

            public void stop() { Cf.list(tasksListener, inspectListener).forEach(PoolListener::stop); }
        };
    }
}
