package ru.yandex.chemodan.queller.worker;

import org.joda.time.Instant;
import org.springframework.amqp.core.MessageProperties;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.bazinga.PgOnetimeUtils;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.queller.celery.QuellerQueues;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.celery.job.CeleryOnetimeJobConverter;
import ru.yandex.chemodan.queller.rabbit.RabbitPool;
import ru.yandex.chemodan.queller.rabbit.RoutedMessage;
import ru.yandex.chemodan.queller.support.BenderJsonMessageConverter;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Instrument;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author dbrylev
 */
public class CeleryTaskManager implements MonicaContainer {

    private static final Logger logger = LoggerFactory.getLogger(CeleryTaskManager.class);

    private final RabbitPool rabbits;
    private final BenderJsonMessageConverter<CeleryJob> converter;

    @MonicaMetric
    @GroupByDefault
    private final Instrument submits = new Instrument();

    public CeleryTaskManager(RabbitPool rabbits) {
        this.rabbits = rabbits;
        this.converter = new BenderJsonMessageConverter<>(CeleryJob.class);
    }

    public FullJobId submit(CeleryOnetimeTask task) {
        Option<String> ycrid = YandexCloudRequestIdHolder.getO();

        if (!task.getParametersTyped().ycrid.isPresent() && ycrid.isPresent()) {
            task.setParameters(task.getParametersTyped().withYcrid(ycrid.get()));
        }
        OnetimeJob onetimeJob = PgOnetimeUtils.makeJob(task, Instant.now(), Option.empty(), 0);

        CeleryJob celeryJob = CeleryOnetimeJobConverter.convertToCelery(onetimeJob, false);
        submit(celeryJob);

        return onetimeJob.getId();
    }

    public void submit(CeleryJob celeryJob) {
        submit(celeryJob, true);
    }

    public void submit(CeleryJob celeryJob, boolean confirmed) {
        RoutedMessage message = new RoutedMessage(
                converter.toMessage(celeryJob, new MessageProperties()),
                QuellerQueues.SUBMIT_EXCHANGE.getName(),
                Option.of(QuellerQueues.SUBMIT_QUEUE.getName()));

        submits.measure(() -> rabbits.sendMessage(message, confirmed));

        logger.info("Task {} submitted to RabbitMQ. ycrid={}", celeryJob.forLog(), celeryJob.getYcrid());
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("celery", new MetricName("celery", "taskManager"), "Celery java tasks manager");
    }
}
