package ru.yandex.chemodan.boot;

import org.apache.commons.lang3.SystemUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ConfigurableApplicationContext;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.boot.admin.ChemodanAdminDaemonContextConfiguration;
import ru.yandex.chemodan.log.Log4jHelper;
import ru.yandex.chemodan.log.StdOutErrLogger;
import ru.yandex.chemodan.util.AppNameHolder;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.LogRotation;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.main.MainSupport;
import ru.yandex.misc.property.load.strategy.PropertiesLoadStrategy;
import ru.yandex.misc.spring.ApplicationContextUtils;
import ru.yandex.misc.version.AbstractVersion;
import ru.yandex.misc.version.AppName;

/**
 * @author vavinov
 */
public abstract class ChemodanMainSupport extends MainSupport {
    private static final Logger logger = LoggerFactory.getLogger(ChemodanMainSupport.class);

    @Value("${startServices:-true}")
    private boolean startServices;

    @Override
    protected void doMain(String[] args) throws Exception {
        AppNameHolder.set(applicationName());
        super.doMain(args);
    }

    public abstract AppName applicationName();

    public abstract ListF<Class<?>> applicationSpecificContextPath();

    @Override
    protected final boolean isCorbaSupported() {
        // zero more projects with this method override and I'll just kill myself
        return false;
    }

    /**
     * disabled default init with log4j
     */
    protected void initializeLoggerPreliminary() { }

    @Override
    protected void initializeAppSpecific() throws Exception {
        String m = applicationName() + " " + DiskAppVersion.VERSION.toString()
                + " started in " + EnvironmentType.getActive() + " mode.";

        String separator = StringUtils.repeat("*", m.length());
        logger.info("");
        logger.info(separator);
        logger.info(StringUtils.capitalize(m));
        logger.info(separator);
        logger.info("");
        logger.info("");
    }

    @Override
    public void initializeLogger() {
        configureLogger();
        if (isDeveloperHost()) {
            Log4jHelper.addAppender(Log4jHelper.consConsoleAppender(), "STDOUT", Level.INFO, false, true);
            StdOutErrLogger.redirectSystemOutErrToLog();
        } else if (isDaemon()) {
            StdOutErrLogger.redirectSystemOutErrToLog();
            LogRotation.installSignalHandler();
        }
    }

    private static boolean isDeveloperHost() {
        return EnvironmentType.getActive() == EnvironmentType.DEVELOPMENT
                && (SystemUtils.IS_OS_MAC || SystemUtils.IS_OS_WINDOWS);
    }

    @Override
    protected void postProcessApplicationContextBeforeRefresh(ConfigurableApplicationContext context) {
        super.postProcessApplicationContextBeforeRefresh(context);
        ApplicationContextUtils.registerSingleton(context, "appName", applicationName());
    }

    @Override
    protected final boolean isStartServices() {
        return startServices;
    }

    @Override
    protected PropertiesLoadStrategy propertiesLoadStrategy() {
        return new ChemodanPropertiesLoadStrategy(applicationName(), false);
    }

    @Override
    @SuppressWarnings("rawtypes")
    protected final Class[] applicationContextPath() {
        return Cf.<Class>list()
                .plus1(ChemodanInitContextConfiguration.class)
                .plus(isDaemon()
                        ? Cf.list(getAdminDaemonContextConfiguration())
                        : Cf.list())
                .plus(applicationSpecificContextPath())
                .toArray(Class.class);
    }

    protected Class<?> getAdminDaemonContextConfiguration() {
        return ChemodanAdminDaemonContextConfiguration.class;
    }

    @Override
    protected void configureLogger() {
        Log4jHelper.rootLoggerBuilder().appName(applicationName()).build();
        Log4jHelper.accessLoggerBuilder().appName(applicationName()).build();
    }

    @Override
    protected AbstractVersion getVersion() {
        return DiskAppVersion.VERSION;
    }

    @Override
    protected boolean isDaemon() {
        return true;
    }

    @Override
    protected boolean needToLogInitMessagesAfterRotate() {
        return false;
    }

}
