package ru.yandex.chemodan.boot;

import java.lang.reflect.Field;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.monica.BucketsFromStatisticsDataExtractor;
import ru.yandex.chemodan.monica.DiskGraphiteMetricsRegistry;
import ru.yandex.chemodan.util.yasm.MonicaYasmContextConfiguration;
import ru.yandex.commune.alive2.location.Location;
import ru.yandex.commune.alive2.location.LocationType;
import ru.yandex.commune.alive2.location.LocationTypeCondition;
import ru.yandex.commune.monica.MonicaClientBaseContextConfiguration;
import ru.yandex.commune.monica.MonicaMyServletContextConfiguration;
import ru.yandex.commune.monica.SolomonMyServletContextConfiguration;
import ru.yandex.commune.monica.admin.MonicaAdminContextConfiguration;
import ru.yandex.commune.monica.collect.MonicaZkHttpClientContextConfiguration;
import ru.yandex.commune.monica.pusher.FieldValues;
import ru.yandex.commune.monica.pusher.FieldValuesExtractor;
import ru.yandex.commune.monica.pusher.MonicaFieldValuesExtractorRegistry;
import ru.yandex.commune.monica.solomon.MonicaSnapshotConverterForSolomon;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.env.Environment;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.exec.ProcessUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.MonicaConfiguration;
import ru.yandex.misc.monica.core.blocks.InstrumentedData;
import ru.yandex.misc.monica.core.blocks.StatisticData;
import ru.yandex.misc.monica.core.name.FullMetricName;
import ru.yandex.misc.monica.core.name.LocalMetricNamespace;
import ru.yandex.misc.monica.metrics.MonicaLog4jMetricsContextConfiguration;
import ru.yandex.misc.monica.solomon.MonicaSolomonConfiguration;
import ru.yandex.misc.monica.solomon.SensorFilter;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.version.AppName;

@Configuration
@Import({
        MonicaClientBaseContextConfiguration.class,
        ChemodanMonicaConfigurtion.HttpClient.class,
        MonicaMyServletContextConfiguration.class,
        SolomonMyServletContextConfiguration.class,
        MonicaLog4jMetricsContextConfiguration.class,
        MonicaYasmContextConfiguration.class
})
public class ChemodanMonicaConfigurtion {

    private static final Logger logger = LoggerFactory.getLogger(ChemodanMonicaConfigurtion.class);

    @Bean
    public MonicaConfiguration monicaConfiguration(
            EnvironmentType environmentType,
            AppName appName,
            Location myLocation,
            DiskGraphiteMetricsRegistry graphiteMetricsRegistry,
            BucketsFromStatisticsDataExtractor bucketsFromStatisticsDataExtractor,
            @Value("${monica.tick.seconds}")
            int monicaTickSeconds)
    {
        String dcName = Environment.isDeveloperNotebook() ? "unknown" : myLocation.dcName.getOrElse("unknown");

        String conductorGroup = Environment.isDeveloperNotebook() ? "unknown" : myLocation.groupName.getOrElse("unknown");


        Field field;
        try {
            //TODO: make it possible to add extractors without hacks

            field = MonicaSnapshotConverterForSolomon.class.getDeclaredField("pusherConfiguration");
            field.setAccessible(true);
            MonicaFieldValuesExtractorRegistry pusherConfiguration = (MonicaFieldValuesExtractorRegistry) field.get(null);

            pusherConfiguration.register("TIME_BUCKET", StatisticData.class, new FieldValuesExtractor<StatisticData>() {
                @Override
                public FieldValues apply(FullMetricName fullMetricName, StatisticData o) {
                    return bucketsFromStatisticsDataExtractor.extract(o);
                }
            });

            pusherConfiguration.register("TIME_BUCKET", InstrumentedData.class, new FieldValuesExtractor<InstrumentedData>() {
                @Override
                public FieldValues apply(FullMetricName fullMetricName, InstrumentedData o) {
                    return bucketsFromStatisticsDataExtractor.extract(o.statisticData());
                }
            });

        } catch (Throwable e) {
            logger.error("Failed to create bucket extractor: {}", e);
            ExceptionUtils.throwIfUnrecoverable(e);
        }


        return new MonicaConfiguration(new LocalMetricNamespace(
                environmentType.getValue(),
                appName.serviceName(),
                appName.appName(),
                conductorGroup,
                dcName,
                HostnameUtils.localHostname(),
                ProcessUtils.getPid()
        )){
            @Override
            protected Duration getDefaultTick() {
                return Duration.standardSeconds(monicaTickSeconds);
            }

            @Override
            public MonicaSolomonConfiguration getSolomonConfiguration() {
                return new MonicaSolomonConfiguration((name) -> {
                    String metricNameStr = name.asList().mkString(".");
                    return graphiteMetricsRegistry
                            .getPatterns()
                            .exists(p -> p.matches(metricNameStr));
                }, SensorFilter.NO_FILTER);
            }
        };
    }

    @Configuration
    @Import({
            MonicaZkHttpClientContextConfiguration.class,
            MonicaAdminContextConfiguration.class,
    })
    @LocationTypeCondition(LocationType.CONDUCTOR)
    public static class HttpClient {
    }
}
