package ru.yandex.chemodan.boot;

import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.commune.alive2.location.DcUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.env.Environment;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.property.load.strategy.PropertiesBuilder;
import ru.yandex.misc.property.load.strategy.PropertiesLoadStrategy;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.version.AppName;

/**
 * @author akirakozov
 */
public class ChemodanPropertiesLoadStrategy implements PropertiesLoadStrategy {
    private static final Logger logger = LoggerFactory.getLogger(ChemodanPropertiesLoadStrategy.class);

    private static final String DEFAULT_PROP_FILE_SUFFIX = "default";

    private static final String TESTS_PROP_FILE_SUFFIX = "tests";

    private static final ListF<String> PROPERTY_FILE_SUFFIXES = Cf.list(EnvironmentType.values())
            .map(Enum::name)
            .map(String::toLowerCase)
            .plus(DEFAULT_PROP_FILE_SUFFIX, TESTS_PROP_FILE_SUFFIX);

    private static final Pattern2 PROPERTIES_FILE_PATTERN = Pattern2.compile(
            String.format("app-([^/]+?)-(?:%s)\\.properties$", StringUtils.join(PROPERTY_FILE_SUFFIXES, "|"))
    );

    private static final String COMPONENT_PROPS_PATH = "/ru/yandex/chemodan/app-";
    private static final String ENV_PREFIX = "disk_";

    private final AppName appName;
    private final boolean forTests;

    public ChemodanPropertiesLoadStrategy(AppName appName, boolean forTests) {
        this.appName = appName;
        this.forTests = forTests;
    }

    @Override
    public void load(PropertiesBuilder t, EnvironmentType envType) {
        t.set("app.name", appName.appName());
        t.set("service.name", appName.serviceName());
        t.setDefault(EnvironmentType.YANDEX_ENVIRONMENT_TYPE_PROPERTY, envType.getValue());

        loadForEnvironment(envType, "classpath:/ru/yandex/chemodan/global/application-", t::include, Option.empty());
        Option<String> secondaryEnvironmentTypeO = EnvironmentType.getActiveSecondary();
        findPropAppNames()
                .minus1(appName.appName())
                .toList()
                .plus1(appName.appName())
                .forEach(app -> loadForEnvironment(envType,
                        "classpath:" + COMPONENT_PROPS_PATH + app + "-",
                        t::includeIfExists, secondaryEnvironmentTypeO)
                );

        // host-specific
        t.includeIfExists("/etc/yandex/" + appName.serviceName() + "/" + appName.appName() + "/application.properties");

        if (Environment.isDeveloperNotebook()) {
            t.includeIfExists("classpath:application.properties");
        }

        // XXX: move to separate class
        if ("dataapi".equals(appName.serviceName())) {
            // include custom properties for memcached
            // different data centers has different properties
            t.includeIfExists("/etc/yandex/dataapi/profile-memcached.properties");
        }

        // usually set in java-with-classpath.sh
        t.includeSystemProperties();
        t.includeCmdLineProperties();

        copyCutCurrentDcPrefix(t);
        loadSystemEnvVariables(t, ENV_PREFIX);

        t.copyCutPrefix(appName.appName() + ".");
        t.copyCutPrefix("service_" + appName.serviceName() + "_all.");
        if (appName.appName().contains("-")) {
            String component = appName.appName().split("-")[0];
            t.copyCutPrefix(component + "_all.");
        }

        for (String context : GlobalContexts.contexts) {
            t.copyCutSuffix("@" + context);
        }

        t.dump();

        // Load secret properties after all to avoid dumping

        // secret properties (moved to server by salt)
        t.includeIfExists("/etc/yandex/" + appName.serviceName()
                + "/" + appName.appName() + "/application-secret.properties");

        // secret properties
        t.includeIfExists("classpath:/ru/yandex/chemodan/application-secret-"
                + envType.name().toLowerCase()
                + ".properties");
    }

    /**
     *
     * methods performs same normalisation for env variables and our properties keys and tries to match them.
     * If keys are equals then normalised key will be replaced with original
     *
     */
    private void loadSystemEnvVariables(PropertiesBuilder t, String prefix) {
        Map<String, String> env = normaliseSysEnvs(System.getenv(), prefix);
        Map<String, String> props = normaliseProperties(t.getProperties());
        Map<String, String> result = env.entrySet().stream()
                .collect(Collectors.toMap(
                        e -> props.getOrDefault(e.getKey(), e.getKey()),
                        e -> e.getValue()));
        t.include(result);
    }

    private static Map<String, String> normaliseSysEnvs(Map<String, String> envs, String prefix) {
        return envs.entrySet().stream()
                .filter(s -> s.getKey().toLowerCase().startsWith(prefix))
                .collect(Collectors.toMap(
                        e -> normaliseSysEnvKey(cutPrefix(e.getKey(), prefix)),
                        e -> e.getValue()));
    }

    private static String cutPrefix(String key, String prefix) {
        return key.substring(prefix.length(), key.length());
    }

    private static String normaliseSysEnvKey(String key) {
        return key.replaceAll("_", ".").toLowerCase();
    }

    /**
     * Our properties contains "-" and "." which are not allowed in env variables
     * so here we build map with key -> as normalised key and value as original key of property
     */
    private static Map<String, String> normaliseProperties(Map<?, ?> props) {
        return props.entrySet().stream()
                .collect(Collectors.toMap(
                        e -> e.getKey().toString().replaceAll("_|-", ".").toLowerCase(),
                        e -> e.getKey().toString(),
                        (p1, p2) -> p1));
    }

    static SetF<String> findPropAppNames() {
        try {
            ListF<Resource> resources = Cf.x(
                    new PathMatchingResourcePatternResolver()
                            .getResources("classpath*:" + COMPONENT_PROPS_PATH + "*.properties")
            );
            return resources.filterMap(r -> PROPERTIES_FILE_PATTERN.findFirstGroup(r.getFilename()))
                    .unique();
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private void loadForEnvironment(EnvironmentType environmentType,
                                    String pathPrefix, Consumer<String> consumer, Option<String> secondaryEnvironmentO) {
        consumer.accept(pathPrefix + DEFAULT_PROP_FILE_SUFFIX + ".properties");
        if (environmentType == EnvironmentType.PRESTABLE || environmentType == EnvironmentType.QA) {
            loadAllLevelEnvironments(EnvironmentType.PRODUCTION, pathPrefix, consumer, secondaryEnvironmentO);
        }
        loadAllLevelEnvironments(environmentType, pathPrefix, consumer, secondaryEnvironmentO);

        if (forTests) {
            consumer.accept(pathPrefix + TESTS_PROP_FILE_SUFFIX + ".properties");
        }
    }

    private void loadAllLevelEnvironments(EnvironmentType environmentType,
            String pathPrefix, Consumer<String> consumer, Option<String> secondaryEnvironmentO) {
        consumer.accept(pathPrefix + environmentType.getValue() + ".properties");
        secondaryEnvironmentO.ifPresent(secondaryEnvironment -> consumer.accept(pathPrefix + environmentType.getValue() + "-" +
                secondaryEnvironment.toLowerCase().replace('_', '-') + ".properties"));
    }

    private void copyCutCurrentDcPrefix(PropertiesBuilder t) {
        if (!Environment.isDeveloperNotebook()) {
            logger.info("Not cutting DC prefix because current host is developer's workstation");
            return;
        }

        Option<String> currentDc = DcUtils.getCurrentDcDirtyO();
        if (!currentDc.isPresent()) {
            logger.warn("Could not determine DC by letter for host: {}", HostnameUtils.localHostname());
            return;
        }

        t.copyCutPrefix(String.format("dc.%s.", currentDc.get()));
    }

}
