package ru.yandex.chemodan.boot.value;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.FieldX;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class OverridableValuePostProcessor implements BeanFactoryPostProcessor, BeanPostProcessor {
    private ConfigurableListableBeanFactory beanFactory;

    private ValueOverrideHelper valueOverrideHelper;

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        this.valueOverrideHelper = new ValueOverrideHelper(beanFactory);
        this.beanFactory = beanFactory;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        valueOverrideHelper.process(bean, beanName);
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    private class ValueOverrideHelper {
        final MapF<String, String> beanNameToPrefix;

        ValueOverrideHelper(ConfigurableListableBeanFactory beanFactory) {
            this.beanNameToPrefix = Cf.list(beanFactory.getBeanDefinitionNames())
                    .zipWith(beanFactory::getBeanDefinition)
                    .filterMap(OverridableValuePostProcessor::toNameAndPrefixO)
                    .toTuple2List(tuple -> tuple)
                    .toMap();
        }

        void process(Object bean, String beanName) {
            String prefix = beanNameToPrefix.getOrElse(beanName, "");
            setOverridableValues(bean, prefix);
            setOverridableValuePrefix(bean, prefix);
        }

        void setOverridableValues(Object bean, String prefix) {
            ClassX.wrap(bean.getClass())
                    .getAllDeclaredInstanceFields()
                    .filter(f -> f.hasAnnotation(OverridableValue.class))
                    .zipWith(f -> f.getAnnotation(OverridableValue.class))
                    .map2(OverridableValue::value)
                    .filterBy2(v -> v.startsWith("${"))
                    .map2(placeholder -> resolvePlaceholder(placeholder, prefix))
                    .forEach((field, value) -> setFieldValue(bean, field, value));
        }

        void setOverridableValuePrefix(Object bean, String prefix) {
            if (bean instanceof OverridableValuePrefixAware) {
                ((OverridableValuePrefixAware) bean).setOverridableValuePrefix(prefix);
            }
        }

        private void setFieldValue(Object bean, FieldX field, String value) {
            Object obj = beanFactory.getTypeConverter()
                    .convertIfNecessary(value, field.getType().getClazz());
            field.setAccessible(true);
            field.set(bean, obj);
        }

        String resolvePlaceholder(String placeholder, String prefix) {
            try {
                return resolvePlaceholder(placeholder.replaceAll("^\\$\\{", "\\$\\{" + prefix + "."));
            } catch (RuntimeException e) {
                return resolvePlaceholder(placeholder);
            }
        }

        String resolvePlaceholder(String placeholder) {
            return beanFactory.resolveEmbeddedValue(placeholder);
        }
    }

    static Option<Tuple2<String, String>> toNameAndPrefixO(Tuple2<String, BeanDefinition> nameAndBeanDefinition) {
        return toNameAndPrefixO(nameAndBeanDefinition.get1(), nameAndBeanDefinition.get2());
    }

    static Option<Tuple2<String, String>> toNameAndPrefixO(String name, BeanDefinition definition) {
        if (!(definition instanceof AnnotatedBeanDefinition)) {
            return Option.empty();
        }

        AnnotatedBeanDefinition abd = (AnnotatedBeanDefinition) definition;
        return Option.ofNullable(abd.getFactoryMethodMetadata())
                .filterMap(metadata ->
                        Option.ofNullable(metadata.getAllAnnotationAttributes(OverridableValuePrefix.class.getName()))
                )
                .filterMap(map -> Option.ofNullable(map.getFirst("value")))
                .map(Object::toString)
                .map(prefix -> new Tuple2<>(name, prefix));
    }
}
