package ru.yandex.chemodan.cache;

import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Gauge;
import ru.yandex.misc.monica.core.blocks.Meter;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author dbrylev
 */
public class MeteredCache<K, V> implements MonicaContainer {

    public final Cache<K, V> cache;

    @GroupByDefault
    @MonicaMetric
    private final Meter hits = new Meter();

    @GroupByDefault
    @MonicaMetric
    private final Meter outdated = new Meter();

    @GroupByDefault
    @MonicaMetric
    private final Gauge<Long> size;

    public MeteredCache(int size, int concurrency, Duration ttl) {
        this(CacheBuilder.<K, V>newBuilder()
                .concurrencyLevel(concurrency)
                .maximumSize(size)
                .expireAfterWrite(ttl.getMillis(), TimeUnit.MILLISECONDS)
                .build());
    }

    public MeteredCache(Cache<K, V> cache) {
        this.cache = cache;
        this.size = Gauge.cons(cache::size, Long.class);
    }

    public static <K, V> MeteredCache<K, V> noCache() {
        return new MeteredCache<>(CacheBuilder.newBuilder().maximumSize(0).build());
    }

    public void incHits() {
        hits.inc();
    }

    public void incOutdated() {
        outdated.inc();
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("cache", new MetricName("cache", instanceName), instanceName);
    }

    public Option<V> getO(K key) {
        return Option.ofNullable(cache.getIfPresent(key));
    }

    public void put(K key, V value) {
        cache.put(key, value);
    }

    public void invalidate(K key) {
        cache.invalidate(key);
    }

    public long size() {
        return cache.size();
    }
}
