package ru.yandex.chemodan.cache;

import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.ParametersAreNonnullByDefault;

import lombok.Builder;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Value;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.function.Function0;
import ru.yandex.misc.cache.CacheBase;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.worker.DelayingWorkerThread;

@ParametersAreNonnullByDefault
public class TimeLimitedCacheFast<K, V> extends CacheBase<K, V> {

    private final Worker worker;
    private final Function0<Integer> ttl;
    private final ConcurrentHashMap<K, CacheEntryValue<V>> cache;

    private Instant checkTime = Instant.now();

    @Builder
    public TimeLimitedCacheFast(Function0<Integer> ttl, String cacheName, float loadFactor, int initialCapacity) {
        Validate.isTrue(StringUtils.isNotEmpty(cacheName));
        this.cache = new ConcurrentHashMap<>(initialCapacity, loadFactor);
        this.ttl = ttl;
        worker = new Worker(cacheName, ObjectUtils.min(Duration.standardMinutes(ttl.apply()), Duration.standardMinutes(1)));
        worker.start();
    }

    @Value
    class Worker extends DelayingWorkerThread {

        private final Duration checkPeriod;

        public Worker(String cacheName, Duration checkPeriod) {
            super("cache-gc-" + cacheName);
            this.checkPeriod = checkPeriod;
        }

        @Override
        protected long delayBetweenExecutionsMillis() {
            return checkPeriod.getMillis();
        }

        @Override
        protected void executePeriodically() {
            checkTime = Instant.now();
            Instant expTime = checkTime.minus(Duration.standardMinutes(ttl.apply()));
            cache.entrySet().removeIf(e -> isExpired(e.getValue(), expTime));
        }
    }

    @Data
    @EqualsAndHashCode(of = "createInstant")
    private static class CacheEntryValue<V> {
        private final V value;
        private long createInstant;

        public CacheEntryValue(V value) {
            this.value = value;
            this.createInstant = System.currentTimeMillis();
        }

    }

    private boolean isExpired(CacheEntryValue<V> entry, Instant instant) {
        if (instant.isAfter(entry.createInstant)) return true;
        return false;
    }

    /**
     * Overrided parent for reducing generating garbage
     */
    @Override
    public Optional<V> getFromCache(K key) {
        CacheEntryValue<V> entry = getFromCacheInternal(key);
        return Optional.ofNullable(entry == null ? null : entry.value);
    }

    @Override
    public Optional<V> getFromCache(K key, Function0<Optional<V>> populateClosure) {
        CacheEntryValue<V> entry = getFromCacheInternal(key);

        if (entry != null) {
            return Optional.of(entry.value);
        } else {
            Optional<V> value = populateClosure.apply();
            value.ifPresent(v -> putInCache(key, value.get()));
            return value;
        }
    }

    private CacheEntryValue<V> getFromCacheInternal(K key) {
        CacheEntryValue<V> cacheEntryValue = cache.get(key);
        if (cacheEntryValue != null) {
            cacheEntryValue.createInstant = checkTime.getMillis();
        }
        return cacheEntryValue;
    }

    @Override
    public void putInCache(K key, V value) {
        cache.put(key, new CacheEntryValue<V>(value));
    }

    @Override
    public void removeFromCache(K key) {
        cache.remove(key);
    }

    @Override
    public void flush() {
        cache.clear();
    }

    @Override
    public void destroy() {
        cache.clear();
        worker.stopGracefully();
    }

    public long size() {
        return cache.size();
    }

} //~
