package ru.yandex.chemodan.cloud.auth;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import javax.servlet.http.HttpServletRequest;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.cloud.auth.config.PlatformClient;
import ru.yandex.chemodan.util.exception.AccessForbiddenException;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Pattern2;

public class ScopesFetcher {

    private static final Logger logger = LoggerFactory.getLogger(ScopesFetcher.class);

    public static final String PROFILE = "profile";
    public static final String SCOPE_PREFIX =  "cloud_api.profile:generic.";
    public static final ListF<String> READONLY_HTTP_METHODS = Cf.list("GET");
    private static final Pattern2 REQUEST_OBJ_SCOPE_RX = Pattern2.compile("personality/profile/(.*)");

    /**
     *     Доступом к динамическим объектам API Персонализации.
     *     Проверяет наличие у клиентов скоупов вида "cloud_api.profile:generic.<resource_path>[.read|.write]".
     *     Для доступа к методам GET, HEAD, OPTIONS достаточно скоупа с суффиксом ".read".
     *     Для доступа ко всем остальным методам, необходим скоуп с суффиксом ".write".
     *     Для полного доступа необходимы оба скоупа.
     *     Например:
     *         Допустим у нас есть ресурс-коллекция "/personality/profile/videosearch/likes"
     *         и ресурс-объект "/personality/profile/videosearch/likes/{id}".
     *         Для readonly доступа к ресурам достаточно чтобы клиент имел скоуп
     *         "cloud_api.profile:generic.videosearch.likes.read".
     *         Для доступа на запись достаточно скоупа "cloud_api.profile:generic.videosearch.likes.write".
     *         Для полного доступа потребуются оба скоупа: "cloud_api.profile:generic.videosearch.likes.read"
     *         и "cloud_api.profile:generic.videosearch.likes.write".
     */
    public List<String> getRequiredScopes(HttpServletRequest request, ListF<String> availableScopes) {
        Option<String> profile =
                REQUEST_OBJ_SCOPE_RX.findFirstGroup(request.getRequestURI());
        if (!profile.isPresent()) {
            // we should not check rest urls:
            return availableScopes;
        }
        String prof = profile.get();

        String method = request.getMethod();
        StringBuilder path = new StringBuilder(SCOPE_PREFIX);
        Optional<String> first = Arrays.stream(prof.split("/")).map(p ->
                path.append(p).append(".").toString() + (READONLY_HTTP_METHODS.containsTs(method) ? "read" : "write")
        ).filter(availableScopes::containsTs).findAny();
        return Option.x(first);
    }

    public void checkScopes(WebRequest request, PlatformClient availableScopes) {
        if (getRequiredScopes(request.getHttpServletRequest(), availableScopes.getOauthScopes()).isEmpty()) {
            logger.error("Access check Failed: availableScopes: {}, {}",
                    availableScopes,
                    request.getHttpServletRequest().getRequestURI());
            throw new AccessForbiddenException("Required scopes was not found: " + availableScopes.getOauthScopes());
        }
    }

}
