package ru.yandex.chemodan.cloud.auth;

import java.util.List;

import lombok.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.cloud.auth.providers.PlatformAuthenticationProvider;
import ru.yandex.chemodan.util.exception.AccessForbiddenException;
import ru.yandex.commune.a3.action.intercept.ActionDispatcherInterceptor;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@Value
public class SecurityPlatformDispatcher implements ActionDispatcherInterceptor {

    private final static Logger logger = LoggerFactory.getLogger(SecurityPlatformDispatcher.class);

    public final static String PLATFORM_ATTRIBUTE = "__platform";

    public final static String UID_PARAMETER = "__uid";

    public final DynamicProperty<String> securityCheckPrefix =
            new DynamicProperty<>("platform.security.checkPrefix", "/platform");

    public final DynamicProperty<String> securityHeader =
            new DynamicProperty<>("platform.security.header", "");

    private final List<PlatformAuthenticationProvider> providerList;
    private final ScopesFetcher scopesFetcher;

    @Override
    public int getOrder() {
        return 0;
    }

    @Override
    public void beforeDispatch(WebRequest webRequest, ActionInvocationContext invocationContext) {
        if (checkSecurity(webRequest)) {
            checkRequest(webRequest);
        } else {
            checkEmbedded(webRequest);
        }
    }

    private void checkRequest(WebRequest webRequest) {
        providerList.stream().filter(provider -> {
            return provider.support(webRequest).map(sup -> {
                sup.getUser().ifPresent(user -> webRequest.getParameters().put(UID_PARAMETER, Cf.list(user)));
                Option<PlatformUserDetails> auth = provider.auth(webRequest, sup);
                return auth.map(au -> {
                    webRequest.getHttpServletRequest().setAttribute(PLATFORM_ATTRIBUTE, au);
                    return true;
                }).getOrElse(false);
            }).getOrElse(false);
        }).findAny().orElseThrow(() -> new AccessForbiddenException("Access denied"));
    }

    private void checkEmbedded(WebRequest webRequest) {
        Option.ofNullable(webRequest.getHttpServletRequest().getAttribute(PLATFORM_ATTRIBUTE))
                .forEach(attribute -> scopesFetcher.checkScopes(webRequest, ((PlatformUserDetails) attribute).getPlatformClient()));
    }

    private boolean checkSecurity(WebRequest webRequest) {
        String headerName = securityHeader.get();
        if (StringUtils.isNotEmpty(headerName)) {
            Option<String> header = webRequest.getHeader(headerName);
            if (header.isPresent()) {
                return true;
            }
        }
        String checkPrefix = securityCheckPrefix.get();
        String pathInfo = webRequest.getHttpServletRequest().getRequestURI();
        if (StringUtils.isNotEmpty(checkPrefix) && StringUtils.isNotEmpty(pathInfo)) {
            return pathInfo.startsWith(checkPrefix);
        }
        return false;
    }
}
