package ru.yandex.chemodan.cloud.auth.config;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.Setter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.cloud.auth.PlatformAuthentication;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.ZkWatcher;
import ru.yandex.commune.zk2.client.Zk;
import ru.yandex.commune.zk2.primitives.observer.ZkPathObserver;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderParser;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.lang.Validate;

/**
 * /cloud-api/production/auth
 *
 * yanews
 */
@Getter
@Setter
public class ZkPlatformClientRepository extends ZkPathObserver
        implements PlatformClientRepository
{

    private final static BenderParser<PlatformClient> benderParser = Bender.parser(PlatformClient.class);

    private volatile MapF<String, PlatformClient> clientsByToken = Cf.map();
    private volatile MapF<String, PlatformClient> clientsByTvm2 = Cf.map();

    public ZkPlatformClientRepository(ZkPath zkPlatformPath) {
        super(zkPlatformPath);
    }

    @Override
    public Option<PlatformClient> findByCredentials(PlatformAuthentication platformAuthentication) {
        Validate.isTrue(clientsByToken.isNotEmpty() || clientsByToken.isNotEmpty(), "Registry is not initialized");
        switch (platformAuthentication.getCredentialType()) {
            case token:
                return clientsByToken.getO(platformAuthentication.getCredentials());
            case tvm_2_0:
                return clientsByTvm2.getO(platformAuthentication.getCredentials());
            default:
                return Option.empty();
        }
    }

    void loadData(byte[] data) {
        ListF<PlatformClient> registrations =
                benderParser.parseListJson(new ByteArrayInputStreamSource(data));
        clientsByToken = registrations.toMapMappingToKey(this::resolveKey);
        MapF<String, PlatformClient> tvm2 = Cf.hashMap();
        registrations.forEach(r -> r.getTvm2clientIds().forEach(c -> tvm2.put(c, r)));
        clientsByTvm2 = tvm2;
    }

    private String resolveKey(PlatformClient c) {
        return c.getToken().getOrElse(c.getOauthClientId().getOrElse(c.getName().get()));
    }

    public void initialize(Zk zk) {
        super.initialize(zk);
        zk.getData(path, new NodeWatcher()).ifPresent(d -> loadData(d));
    }

    @AllArgsConstructor
    private class NodeWatcher extends ZkWatcher {

        @Override
        protected void onNodeDataChanged(ZkPath eventPath) {
            submitBlockingAction("load-data",
                    () -> zk().getData(eventPath, this).ifPresent(d -> loadData(d)));
        }

        @Override
        protected String name() {
            return "platform-auth-data-watcher";
        }
    }

}
