package ru.yandex.chemodan.cloud.auth.providers;

import lombok.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.cloud.auth.PlatformAuthentication;
import ru.yandex.chemodan.cloud.auth.PlatformUserDetails;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 *     Авторизация для сервисов по токену. Используется только во внутреннем API.
 *
 *     Происходит с помощью обязательного параметра:
 *         * token
 *     И необязательного:
 *         * uid
 *
 *     Если uid нет, то будем искать его в url. Если его и в url нет, то дальнейшая авторизация будет проводиться в
 *     заисимости от значения атрибута auth_user_required в хэндлере.
 *
 *     Параметры для идентификации клиентов можно передавать 2мя способами:
 *         1. В заголовке Authorization в формате Authorization: ClientToken token=<client_token>[;uid=<uid>]
 *         2. В query string запроса.
 *
 *     Примеры:
 *         $ curl  'http://localhost:21859/v1/batch/request?fields=items.code,items.body&__uid=1123132' -H "Content-Type: application/json; charset=utf-8" -X POST -d '{ "items": [ {"method":"GET", "relative_url":"/v1/personality/profile/maps_common/pointshistory"} ] }' -H 'Authorization: ClientToken token=a2f313419bd193e79cad;uid=142581702'
 */
@Value
public class TokenHeaderAuthenticationProvider implements PlatformAuthenticationProvider {

    private final static Pattern2 pattern = Pattern2.compile("^ClientToken (token\\s*=([^;$]+)){0,1};?(uid\\s*=([^;$]+)){0,1};?");

    private final Authenticator authenticator;

    public Option<PlatformAuthentication> support(WebRequest webRequest) {
        HttpServletRequestX httpServletRequest = webRequest.getHttpServletRequest();
        Option<String> token = Option.empty();
        Option<String> uid = Option.empty();
        Option<String> authorization = httpServletRequest.getHeaderO("Authorization");
        if (authorization.isPresent()) {
            Matcher2 matcher = pattern.matcher2(authorization.get());
            if (matcher.matches()) {
                token = matcher.group(2);
                uid = matcher.group(4);
            }
        }
        if (!token.isPresent()) {
            token = webRequest.getParameter("token").getO(0);
        }
        if (!uid.isPresent()) {
            uid = webRequest.getParameter("uid").getO(0);
        }
        if (!token.isPresent()) {
            return Option.empty();
        }
        return Option.of(PlatformAuthentication.builder()
                .credentials(token.get())
                .user(uid)
                .credentialType(PlatformAuthentication.AuthType.token)
                .build());
    }

    @Override
    public Option<PlatformUserDetails> auth(WebRequest webRequest, PlatformAuthentication platformAuthentication) {
        return authenticator.auth(webRequest, platformAuthentication);
    }
}
