package ru.yandex.chemodan.cloud.auth.providers;

import java.util.Optional;

import lombok.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.cloud.auth.PlatformAuthentication;
import ru.yandex.chemodan.cloud.auth.PlatformUserDetails;
import ru.yandex.chemodan.util.tvm.TvmClientInfo;
import ru.yandex.chemodan.util.tvm.TvmClientInfoRegistry;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.inside.passport.tvm2.web.Tvm2Filter;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;

@Value
public class Tvm2AuthenticationProvider implements PlatformAuthenticationProvider {

    private final static String X_UID = "X-Uid";
    private final static Logger logger = LoggerFactory.getLogger(Tvm2AuthenticationProvider.class);

    private final Authenticator authenticator;
    private final Optional<Tvm2> tvm2;

    private final TvmClientInfoRegistry tvmClientInfoRegistry;

    public Option<PlatformAuthentication> support(WebRequest webRequest) {
        if (!tvm2.isPresent()) {
            return Option.empty();
        }
        HttpServletRequestX httpServletRequest = webRequest.getHttpServletRequest();
        Option<String> serviceTicket = httpServletRequest.getHeaderO(TvmHeaders.SERVICE_TICKET);
        Option<String> uid = httpServletRequest.getHeaderO(X_UID);
        if (!serviceTicket.isPresent() || !uid.isPresent()) {
            return Option.empty();
        }
        Option<Integer> clientId;
        Object tvmAttribute = httpServletRequest.getAttribute(Tvm2Filter.TVM_CLIENT_ID_ATTR);
        if (tvmAttribute == null) {
            //clientId =
            Option<TvmClientInfo> tvmClientInfo = fetchClientId(serviceTicket);
            clientId = tvmClientInfo.map(t -> t.tvmClientId);
            if (!checkUserTicket(httpServletRequest.getHeaderO(TvmHeaders.USER_TICKET), uid, tvmClientInfo)) {
                logger.warn("Client ticket validation failed: {}", uid, tvmClientInfo);
                return Option.empty();
            }
        } else {
            clientId = Option.of((Integer) tvmAttribute);
        }

        if (!clientId.isPresent()) {
            return Option.empty();
        }
        return Option.of(PlatformAuthentication.builder()
                .credentials(Integer.toString(clientId.get()))
                .credentialType(PlatformAuthentication.AuthType.tvm_2_0)
                .user(uid).build());
    }

    protected boolean checkUserTicket(Option<String> userTicketO, Option<String> uidO, Option<TvmClientInfo> tvmClientInfo) {
        return tvmClientInfo.filter(t -> !t.allowSetAnyUid)
                .map(t -> checkUserTicketInternal(userTicketO, uidO.get())).getOrElse(true);
    }

    private boolean checkUserTicketInternal(Option<String> ut, String uid) {
        return ut.map(u -> {
            try {
                tvm().checkUserTicketByUid(u, Long.valueOf(uid));
                return true;
            } catch (Exception e) {
                logger.warn("Client ticket validation failed {} {}", uid, ut, e);
                return false;
            }
        }).getOrElse(false);
    }

    private Option<TvmClientInfo> fetchClientId(Option<String> serviceTicket) {
        try {
            CheckedServiceTicket st = tvm().checkServiceTicket(serviceTicket.get());
            if (!st.booleanValue()) {
                logger.warn("Ticket disabled {}", st.getSrc());
                return Option.empty();
            }
            Option<TvmClientInfo> tvmClientInfo = tvmClientInfoRegistry.getO(st.getSrc());
            if (!tvmClientInfo.isPresent()) {
                logger.warn("Ticket not registered in datasync {}", st.getSrc());
                return tvmClientInfo;
            }
            return tvmClientInfo;
        } catch (Exception e) {
            logger.warn("Can't auth using tvm2", e);
            return Option.empty();
        }
    }

    @Override
    public Option<PlatformUserDetails> auth(WebRequest webRequest, PlatformAuthentication platformAuthentication) {
        return authenticator.auth(webRequest, platformAuthentication);
    }

    Tvm2 tvm() {
        return tvm2.get();
    }
}
