package ru.yandex.chemodan.dc.closing;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.alive2.AliveAppRegistrar;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author friendlyevil
 */
public class AliveAppInfoHolderTest {
    @Mock
    private AliveAppRegistrar aliveAppRegistrar;
    private final AliveAppInfo aliveAppInfo = new AliveAppInfo("test_service_name", "test_app_name",
            Instant.now(), "test_version", "127.0.0.1", 1234, "", Option.of("test_dc"));
    private AliveAppInfoHolder aliveAppInfoHolder;
    private static final String STATE_MESSAGE = "dataapi state message";

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        this.aliveAppInfoHolder = new AliveAppInfoHolder(aliveAppRegistrar, aliveAppInfo);
    }

    @Test
    public void testSimpleStateChanging() {
        append(STATE_MESSAGE);
        assertEquals(aliveAppInfoHolder.getState(), STATE_MESSAGE);

        remove(STATE_MESSAGE);
        checkEmptyState();
    }

    @Test
    public void testDoubleAppend() {
        append(STATE_MESSAGE);
        append(STATE_MESSAGE);

        assertEquals(aliveAppInfoHolder.getState(), STATE_MESSAGE);
        remove(STATE_MESSAGE);
        checkEmptyState();
    }

    @Test
    public void testRemoveNonExistMessage() {
        String message = "other message";
        remove(STATE_MESSAGE);

        checkEmptyState();
        append(message);
        assertEquals(aliveAppInfoHolder.getState(), message);
        remove(STATE_MESSAGE);
        assertEquals(aliveAppInfoHolder.getState(), message);
    }

    @Test
    public void testStateWithCrossMessages() {
        String first = "dataapi state message";
        String second = "dataapi state";
        String third = "state";

        append(first);
        append(second);
        append(third);
        assertEquals(aliveAppInfoHolder.getState(), first + ';' + second + ';' + third);

        remove(second);
        assertTrue(aliveAppInfoHolder.stateContains(first));
        assertTrue(aliveAppInfoHolder.stateContains(third));
        assertFalse(aliveAppInfoHolder.stateContains(second));
        assertEquals(aliveAppInfoHolder.getState(), first + ';' + third);

        remove(first);
        assertTrue(aliveAppInfoHolder.stateContains(third));
        assertEquals(aliveAppInfoHolder.getState(), third);

        remove(third);
        checkEmptyState();
    }

    @Test
    public void testStateWithCrossMessages1() {
        String first = "dataapi state message";
        String second = "dataapi state";
        String third = "state";

        append(first);
        append(second);
        assertEquals(aliveAppInfoHolder.getState(), first + ';' + second);

        remove(first);
        assertEquals(aliveAppInfoHolder.getState(), second);

        append(third);
        assertEquals(aliveAppInfoHolder.getState(), second + ';' + third);

        append(first);
        assertEquals(aliveAppInfoHolder.getState(), second + ';' + third + ';' + first);

        remove(first);
        assertEquals(aliveAppInfoHolder.getState(), second + ';' + third);
    }


    private void checkEmptyState() {
        assertTrue(aliveAppInfoHolder.getState().isEmpty());
    }

    private void append(String message) {
        aliveAppInfoHolder.appendToState(message);
        assertTrue(aliveAppInfoHolder.stateContains(message));
    }

    private void remove(String message) {
        aliveAppInfoHolder.removeFromState(message);
        assertFalse(aliveAppInfoHolder.stateContains(message));
    }
}
