package ru.yandex.chemodan.http;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.text.DecimalFormat;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.client.methods.HttpRequestWrapper;
import org.apache.http.protocol.HttpContext;
import org.springframework.util.StreamUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.log.CloseableMdc;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class LoggingHttpInterceptor implements HttpRequestInterceptor, HttpResponseInterceptor {
    private static final String START_TIME_ID = "mpfs.start_time";
    private static final String REQUEST = "mpfs.request";
    private static final SetF<String> IGNORED_HEADERS = Cf.set(HttpHeaders.ACCEPT, HttpHeaders.ACCEPT_ENCODING,
            HttpHeaders.CONNECTION, HttpHeaders.USER_AGENT);
    private static final DecimalFormat msFormat = new DecimalFormat("#0.000");
    private static final Logger requestsLogger = LoggerFactory.getLogger("requests");

    private static final String MPFS_TARGET = "mpfs.requests";
    public static final LoggingHttpInterceptor MPFS_CLIENT_INTERCEPTOR = new LoggingHttpInterceptor(MPFS_TARGET, false);

    private String target;
    private boolean logResponseBody;

    public LoggingHttpInterceptor(String target, boolean logResponseBody) {
        this.target = target;
        this.logResponseBody = logResponseBody;
    }

    @Override
    public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
        context.setAttribute(START_TIME_ID, System.currentTimeMillis());
        if (request instanceof HttpRequestWrapper) {
            request = ((HttpRequestWrapper) request).getOriginal();
        }
        context.setAttribute(REQUEST, request);
    }

    @Override
    public void process(HttpResponse response, HttpContext context) throws HttpException, IOException {
        Long startTime = (Long) context.getAttribute(START_TIME_ID);
        HttpRequest request = (HttpRequest) context.getAttribute(REQUEST);
        Check.notNull(startTime, "LoggingHttpInterceptor is not added as HttpRequestInterceptor");
        Check.notNull(request, "LoggingHttpInterceptor is not added as HttpRequestInterceptor");

        if (requestsLogger.isEnabledFor(Level.INFO)) {
            long elapsedTimeMs = System.currentTimeMillis() - startTime;
            long bytesSent = getBytesSent(request);

            long bytesReceived = response.getEntity().getContentLength();
            // иначе парсилка логов для графиков не работает
            bytesReceived = bytesReceived > 0 ? bytesReceived : 0;

            StringBuilder sb = new StringBuilder();
            sb.append(request.getRequestLine().getMethod());
            sb.append(" \"");
            sb.append(request.getRequestLine().getUri());

            String headers = Cf.wrap(request.getAllHeaders()).filter(x -> !IGNORED_HEADERS.containsTs(x.getName()))
                    .map(x -> "'" + x.getName() + "': '" + x.getValue() + "'").mkString(", ");
            if (!StringUtils.isBlank(headers)) {
                sb.append(" headers:{");
                sb.append(headers);
                sb.append("}");
            }

            if (request instanceof HttpEntityEnclosingRequest) {
                HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
                if (entity != null && entity.isRepeatable()) {
                    try (InputStream content = entity.getContent()) {
                        String body = StreamUtils.copyToString(content, StandardCharsets.UTF_8);
                        sb.append(" body:");
                        sb.append(body);
                    }
                }
            }

            sb.append("\" ");
            sb.append(response.getStatusLine().getStatusCode());
            sb.append(" ");
            sb.append(bytesSent);
            sb.append(" ");
            sb.append(bytesReceived);
            sb.append(" ");
            sb.append(msFormat.format(elapsedTimeMs / 1000d));

            if (logResponseBody && response.getEntity() != null && response.getEntity().isRepeatable()) {
                try (InputStream content = response.getEntity().getContent()) {
                    String body = StreamUtils.copyToString(content, StandardCharsets.UTF_8);
                    sb.append(" response body: '");
                    sb.append(body);
                    sb.append("'");
                }
            }

            try (final CloseableMdc.Instance ignored = CloseableMdc.put("module", "client")) {
                try (final CloseableMdc.Instance ignored2 = CloseableMdc.put("name", target)) {
                    requestsLogger.info(sb.toString());
                }
            }
        }
    }

    private long getBytesSent(HttpRequest request) {
        long bytesSent;
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
            bytesSent = entity != null ? entity.getContentLength() : 0;
        } else {
            bytesSent = 0;
        }
        // иначе парсилка логов для графиков не работает
        bytesSent = bytesSent > 0 ? bytesSent : 0;
        return bytesSent;
    }
}
