package ru.yandex.chemodan.http;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.HandlerWrapper;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.WhatThreadDoes;

/**
 * @author nshmakov
 */
public class YandexCloudRequestIdHandler extends HandlerWrapper {
    public static final String PARENT_RID_PARAM_NAME = "disk-parent-rid";

    private final Function<HttpServletRequest, String> prefixResolver;

    public YandexCloudRequestIdHandler(Function<HttpServletRequest, String> prefixResolver) {
        this.prefixResolver = prefixResolver;
    }

    @Override
    public void handle(String target, Request baseRequest, HttpServletRequest request,
            HttpServletResponse response) throws IOException, ServletException
    {
        // in case of uncaught exception Jetty 9.4 calls handler more than once,
        // but to preserve rid, we MUST generate rid only once per each request
        if (RequestNdcUtil.ndcIsNotStored(request)) {
            // removing NDC and rid completely, because current rid and ycrid are not removed at the end of request
            Ndc.clear();
            WhatThreadDoes.reset();
            RequestIdStack.clearAndPush(getRid(request));
            YandexCloudRequestIdHolder.setAndPushToNdc(getYcrid(request));
            RequestNdcUtil.storeCurrentNdc(request);
        }

        super.handle(target, baseRequest, request, response);

        // rid and ycrid are not removed intentionally to preserve them for logging inside subsequent Jetty calls
    }

    private String getRid(HttpServletRequest request) {
        return Option.ofNullable(request.getParameter(PARENT_RID_PARAM_NAME))
                    .map(s -> String.format("%s-%s", s, RequestIdStack.generateId()))
                    .getOrElse(RequestIdStack.generateId());
    }

    private String getYcrid(HttpServletRequest request) {
        return Option.ofNullable(request.getHeader(CommonHeaders.YANDEX_CLOUD_REQUEST_ID)).getOrElse(generate(request));
    }

    public static String generate(String prefix) {
        String id = Md5.A.consSum(Random2.R.nextBytes(16)).hex();
        return prefix + "-" + id + "-" + HostnameUtils.localHostnameShort();
    }

    public String generate(HttpServletRequest request) {
        return generate(prefixResolver.apply(request));
    }
}
