package ru.yandex.chemodan.http.proxy;

import java.util.concurrent.ThreadLocalRandom;

import lombok.AllArgsConstructor;
import lombok.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.sharpei.UserId;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.alive2.AliveAppsHolder;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author vpronto
 */
@Value
@AllArgsConstructor
public class ProxyManager {
    private static final Logger logger = LoggerFactory.getLogger(ProxyManager.class);

    public static final String SKIP_PROXY = "X-skipProxy";

    public final DynamicProperty<ListF<String>> enabledHosts = new DynamicProperty<>(
            "proxy-forwarded.enabled.hosts", Cf.list()
    );

    public final DynamicProperty<ListF<String>> disabledSourceDc = new DynamicProperty<>(
            "proxy-forwarded.disabled.dc.source", Cf.list()
    );

    public final DynamicProperty<ListF<String>> disabledTargetDc = new DynamicProperty<>(
            "proxy-forwarded.disabled.dc.target", Cf.list()
    );

    public final DynamicProperty<ListF<String>> enabledMethods = new DynamicProperty<>(
            "proxy-forwarded.enabled.methods", Cf.list(HttpMethod.POST.name(), HttpMethod.PUT.name(), HttpMethod.PATCH.name())
    );

    public final DynamicProperty<Boolean> allHostsRedirect = new DynamicProperty<>(
            "proxy-forwarded.enabled.all-hosts", false
    );

    private final MasterDcHostResolver masterDcHostResolver;
    private final ProxyService httpProxy;
    private final AliveAppsHolder aliveAppsHolder;
    private final AliveAppInfo aliveAppInfo;
    private final int retryCount;

    public ProxyManager(MasterDcHostResolver masterDcHostResolver, ProxyService httpProxy,
                        AliveAppsHolder aliveAppsHolder, AliveAppInfo aliveAppInfo) {
        this(masterDcHostResolver, httpProxy, aliveAppsHolder, aliveAppInfo, 0);
    }

    public Option<String> proxy(WebRequest webRequest, UserId dataApiUserId) {
        if (shouldProxy(webRequest)) {
            return proxyInner(webRequest, dataApiUserId);
        }
        return Option.empty();
    }

    public boolean shouldProxy(WebRequest webRequest) {
        return enabledForHost() && enabledForRequest(webRequest) && enabledForMethod(webRequest);
    }

    boolean enabledForHost() {
        return (allHostsRedirect.get() && !disabledSourceDc.get().containsTs(aliveAppInfo.getDc().getOrElse("-"))) ||
                enabledHosts.get().containsTs(aliveAppInfo.getHostname());
    }

    /**
     * Probably also check from 'readFromMaster'
     *
     * boolean readFromMaster(WebRequest webRequest) {
     *         Option<Boolean> isReadFromMaster =
     *                 webRequest.getParameter(READ_FROM_MASTER).firstO().map(Boolean::valueOf);
     *         if (isReadFromMaster.isPresent() && isReadFromMaster.get()) {
     *             return false;
     *         }
     *     }
     */
    boolean enabledForRequest(WebRequest webRequest) {
        Object skipProxy = webRequest.getHttpServletRequest().getAttribute(SKIP_PROXY);
        if (skipProxy != null) {
            return false;
        }
        Option<String> skipHeader = webRequest.getHeader(SKIP_PROXY);
        if (skipHeader.isPresent()) {
            return false;
        }
        return true;
    }

    boolean enabledForMethod(WebRequest webRequest) {

        ListF<String> httpMethod = webRequest.getParameter("http_method")
                .plus(webRequest.getParameter("httpMethod"));
        String method;
        if (httpMethod.isEmpty()) {
            method = webRequest.getHttpServletRequest().getMethod().toUpperCase();
        } else {
            method = httpMethod.first();
        }
        return enabledMethods.get().containsTs(method);

    }

    public Option<String> proxyInner(WebRequest webRequest, UserId dataApiUserId) {
        Option<String> dcs = masterDcHostResolver.resolveMasterHost(dataApiUserId);
        if (disabledForTargetDc(dcs) || !dcs.isPresent()) {
            return Option.empty();
        }

        return resolveHostAndProxyWithRetries(webRequest, dcs.get(), retryCount);
    }

    private Option<String> resolveHostAndProxyWithRetries(WebRequest webRequest, String dc, int retryCount) {
        Option<String> result;
        do {
            result = resolveHostAndProxy(webRequest, dc);
        } while (retryCount-- > 0 && (!result.isPresent() || needRetry(webRequest)));

        if (needRetry(webRequest)) {
//            если не получилось проксировать, то нужно выполнять локально
            return Option.empty();
        }

        return result;
    }

    private boolean needRetry(WebRequest webRequest) {
        int status = webRequest.getHttpServletResponse().getStatus();
//        ретраим только всячиские проблемы с доставкой запроса до нужного хоста. 500 ретраить не нужно
        return status > 500;
    }

    private Option<String> resolveHostAndProxy(WebRequest webRequest, String dc) {
        Option<String> host = resolveHost(dc);
        if (!host.isPresent() || host.isSome(aliveAppInfo.getHostname())) { return Option.empty(); }
        logger.info("Try proxy to {}", host.get());
        HttpProxyContext proxyContext = new HttpProxyContext(webRequest.getHttpServletRequest(),
                webRequest.getHttpServletResponse(),
                host.get());
        return httpProxy.service(proxyContext);
    }

    private boolean disabledForTargetDc(Option<String> dcs) {
        return !dcs.isPresent() || aliveAppInfo.getDc().equals(dcs) || disabledTargetDc.get().containsTs(dcs.get());
    }

    Option<String> resolveHost(String dc) {
        ListF<String> hosts = aliveAppsHolder.aliveApps().filterMap(
                app -> Option.when(
                        app.getServiceName().equals(aliveAppInfo.getServiceName())
                        && app.getAppName().equals(aliveAppInfo.getAppName())
                        && compareAppVersions(app.getVersion(), aliveAppInfo.getVersion())
                        && app.getDc().isSome(dc)
                        && StringUtils.isEmpty(app.getState()),
                        app.getHostname()));
        if (hosts.isEmpty()) {
            return Option.empty();
        }
        String host = hosts.get(ThreadLocalRandom.current().nextInt(hosts.length()));
        return Option.of(host);

    }

    private boolean compareAppVersions(String first, String second) {
        return getMajorVersionPart(first).equals(getMajorVersionPart(second));
    }

//    TODO удалить этот костыль после переезда с железа в деплой (нужен только на вермя переезда)
    private String getMajorVersionPart(String version) {
        int secondDot = version.indexOf('.', version.indexOf('.') + 1);
        if (secondDot > 0) {
            return version.substring(0, secondDot);
        }

        return version;
    }
}
