package ru.yandex.chemodan.http.proxy;

import org.joda.time.Instant;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.sharpei.SimpleUserId;
import ru.yandex.commune.a3.action.WebRequestMock;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.alive2.AliveAppsHolder;

import static ru.yandex.chemodan.http.proxy.ProxyManager.SKIP_PROXY;

public class ProxyManagerTest {

    private final Option<String> masterDc = Option.of("IVA");
    private final AliveAppInfo appName = new AliveAppInfo("test-service", "test", Instant.now(),
            "1", "127.0.0.1", 1, "", Option.of("FOL"));
    private final AliveAppsHolder aliveAppsHolder = new AliveAppsHolder();
    private ProxyManager proxyManager;

    @Before
    public void prepare() {

        aliveAppsHolder.listChanged(Cf.list(createAliveAppEmptyDc(),
                createAliveAppSameDc(),
                createAliveAppAnotherService(),
                createAliveAppAnotherApp(),
                createAliveApp1(),
                createAliveApp2()));
        proxyManager = new ProxyManager(
                userId -> masterDc,
                proxyContext -> Option.of("host"),
                aliveAppsHolder,
                appName);
    }

    @Test
    public void testResolveHosts() {

        Option<String> iva = proxyManager.resolveHost(masterDc.get());
        Assert.assertTrue(iva.isMatch(s -> s.startsWith("right")));
    }

    @Test
    public void testProxy() {

        Option<String> proxy = proxyManager.proxyInner(new WebRequestMock(), new SimpleUserId("123"));
        Assert.assertTrue(proxy.isPresent());
    }

    @Test
    public void testSkipProxy() {
        boolean enabled = proxyManager.enabledForRequest(new WebRequestMock().header(SKIP_PROXY, 1));
        Assert.assertFalse(enabled);
        WebRequestMock webRequestMock = new WebRequestMock();
        webRequestMock.getHttpServletRequest().setAttribute(SKIP_PROXY, 1);
        enabled = proxyManager.enabledForRequest(webRequestMock);
        Assert.assertFalse(enabled);
    }

    @Test
    public void testMethods() {
        boolean enabled = proxyManager.enabledForMethod(new WebRequestMock());
        Assert.assertFalse(enabled);

        enabled = proxyManager.enabledForMethod(WebRequestMock
                .consGet("/api/process_generic_data/?http_method=GET&resource_path=district%2Fpolygons&")
                .param("http_method", "GET"));
        Assert.assertFalse(enabled);
    }

    @Test
    public void testProxySameDc() {
        proxyManager = new ProxyManager(
                userId -> masterDc,
                proxyContext -> Option.of("host"),
                aliveAppsHolder,
                new AliveAppInfo(appName.getServiceName(), appName.getAppName(), Instant.now(),
                        appName.getVersion(), appName.getHostname(), appName.getPid(), appName.getState(), masterDc));

        Option<String> proxy = proxyManager.proxyInner(new WebRequestMock(), new SimpleUserId("123"));
        Assert.assertFalse(proxy.isPresent());
    }

    @Test
    public void testProxyEmptyDc() {
        proxyManager = new ProxyManager(
                userId -> masterDc,
                proxyContext -> Option.of("host"),
                aliveAppsHolder,
                new AliveAppInfo(appName.getServiceName(), appName.getAppName(), Instant.now(),
                        appName.getVersion(), appName.getHostname(), appName.getPid(), appName.getState(), Option.empty()));

        Option<String> proxy = proxyManager.proxyInner(new WebRequestMock(), new SimpleUserId("123"));
        Assert.assertTrue(proxy.isPresent());
    }

    @Test
    public void testProxyEmptyMaster() {
        proxyManager = new ProxyManager(
                userId -> Option.empty(),
                proxyContext -> Option.of("host"),
                aliveAppsHolder,
                new AliveAppInfo(appName.getServiceName(), appName.getAppName(), Instant.now(),
                        appName.getVersion(), appName.getHostname(), appName.getPid(), appName.getState(), masterDc));

        Option<String> proxy = proxyManager.proxyInner(new WebRequestMock(), new SimpleUserId("123"));
        Assert.assertFalse(proxy.isPresent());
    }

    private AliveAppInfo createAliveAppEmptyDc() {
        return createBaseAliveApp("wrong-0", Option.empty());
    }

    private AliveAppInfo createAliveAppSameDc() {
        return createBaseAliveApp("wrong-1", Option.of("MYT"));
    }

    private AliveAppInfo createAliveAppAnotherService() {
        return new AliveAppInfo("another-test-service",
                appName.getAppName(),
                appName.getStartTime(),
                appName.getVersion(),
                "wrong-2",
                appName.getPid(),
                appName.getState(),
                masterDc);
    }

    private AliveAppInfo createAliveAppAnotherApp() {
        return new AliveAppInfo(appName.getServiceName(),
                "another-test",
                appName.getStartTime(),
                appName.getVersion(),
                "wrong-3",
                appName.getPid(),
                appName.getState(),
                masterDc);
    }

    private AliveAppInfo createAliveApp1() {
        return createBaseAliveApp("right-0", masterDc);
    }

    private AliveAppInfo createAliveApp2() {
        return createBaseAliveApp("right-1", masterDc);
    }

    private AliveAppInfo createBaseAliveApp(String hostname, Option<String> dc) {
        return new AliveAppInfo(appName.getServiceName(),
                appName.getAppName(),
                appName.getStartTime(),
                appName.getVersion(),
                hostname,
                appName.getPid(),
                appName.getState(),
                dc);
    }
}
