package ru.yandex.chemodan.log;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.RequestLog;
import org.eclipse.jetty.server.Response;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.chemodan.http.RequestNdcUtil;
import ru.yandex.chemodan.log.utils.ExtraRequestLogFieldsUtil;
import ru.yandex.chemodan.log.utils.TskvEscapeUtils;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.web.servlet.HttpRequestUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @url https://st.yandex-team.ru/CHEMODAN-2769
 * @see ru.yandex.misc.web.servletContainer.jetty.Log4jRequestLog
 * @author ssytnik
 */
public class DiskLog4jRequestLog extends org.eclipse.jetty.util.component.AbstractLifeCycle implements RequestLog {

    public static final String UID_ATTRIBUTE = "request-uid";
    public static final String LOGIN_ATTRIBUTE = "request-login";
    public static final String CLIENT_ID_ATTRIBUTE = "request-client-id";
    public static final String AUTH_SOURCE_ATTRIBUTE = "request-auth-source";

    public static final String CLIENT_DISCONNECTED = "client-was-disconnected";

    private static final ru.yandex.misc.log.mlf.Logger logger =
            ru.yandex.misc.log.mlf.LoggerFactory.getLogger(Log4jHelper.accessLogName);
    private static final ru.yandex.misc.log.mlf.Logger tskvLogger =
            ru.yandex.misc.log.mlf.LoggerFactory.getLogger(Log4jHelper.accessLogTskvName);

    @Override
    public void log(Request req, Response resp) {
        Option<String> ndc = RequestNdcUtil.getNdcStrO(req);
        Option<Ndc.Handle> handle = ndc.map(Ndc::pushReplace);
        try {
            HttpServletRequestX wrappedReq = HttpServletRequestX.wrap(req);
            String agent = wrappedReq.getUserAgent().orElse("-");
            String xForwardedFor = wrappedReq.getHeaderO(CommonHeaders.X_FORWARDED_FOR).orElse("-");
            String xRealIp = wrappedReq.getHeaderO(CommonHeaders.X_REAL_IP).orElse("-");
            String requestTime = TimeUtils.millisecondsToSecondsStringToNow(req.getTimeStamp());

            if (logger.isInfoEnabled()) {
                logNormal(req, resp, agent, xForwardedFor, requestTime);
            }
            logTskv(req, resp, agent, xForwardedFor, xRealIp, requestTime);

        } finally {
            handle.forEach(Ndc.Handle::popSafely);
        }
    }

    private void logNormal(Request req, Response resp,
            String agent, String xForwardedFor, String requestTime)
    {
        logger.info(
                "\"" + req.getMethod() + " " + req.getRequestURI() + "?" + req.getQueryString() + "\" " +
                req.getProtocol() + " \"" + agent + "\" " +
                "\"" + xForwardedFor + "\" " +
                resp.getStatus() + " " + req.getRemoteAddr() + " " +
                req.getContentLength() + " " + resp.getContentCount() + " " +
                requestTime
        );
    }

    private void logTskv(Request req, Response resp,
            String agent, String xForwardedFor, String xRealIp, String requestTime)
    {
        String uid = getUid(req);
        String login = Option.ofNullable(req.getAttribute(LOGIN_ATTRIBUTE)).map(Object::toString).orElse("-");
        String clientId = getClientId(req);
        String authSource =
                Option.ofNullable(req.getAttribute(AUTH_SOURCE_ATTRIBUTE)).map(Object::toString).orElse("-");

        // NB! Field order matters - CHEMODAN-36872
        Tuple2List<String, String> fields = Tuple2List.<String, String>fromPairs(
                "method", req.getMethod(),
                "protocol", req.getProtocol(),
                "request", req.getPathInfo() + (req.getQueryString() == null ? "" : "?" + req.getQueryString()),
                "user_agent", agent,
                "x_forwarded_for", xForwardedFor,
                "x_real_ip", xRealIp,
                "vhost", req.getServerName(),
                "vport", String.valueOf(req.getServerPort()),
                "headers", makeHeadersLogLine(HttpServletRequestX.wrap(req).getHeaders()),
                "ip", req.getRemoteAddr(),
                "status", getStatus(req, resp),
                "request_length", String.valueOf(req.getContentLength()),
                "bytes_sent", String.valueOf(resp.getContentCount()),
                "request_time", requestTime,
                "cookies", "-",
                "referer", "-",
                "uid", uid,
                "login", login,
                "client_id", clientId,
                "auth_source", authSource)
                .plus(ExtraRequestLogFieldsUtil.getFields(req).entries())
                // Экранируем только табуляцию потому что дальше эти поля попадут в TskvLogPatternLayout который заэкранирует все остальное
                // \t там не экранируются чтобы работало разбиение на поля, которое было получено тут
                .map2(value -> TskvEscapeUtils.escapeWithBackslash(value, TskvEscapeUtils.DEFAULT_SYMBOL_ESCAPES_ONLY_TABULATION));

        // XXX: CHEMODAN-36301
        tskvLogger.info(fields.map((key, value) -> key + "=" + value).mkString("\t"));
    }

    public static String getUid(HttpServletRequest req) {
        return Option.ofNullable(req.getAttribute(UID_ATTRIBUTE))
                .map(Object::toString)
                .orElse("-");
    }

    public static String getClientId(HttpServletRequest req) {
        return Option.ofNullable(req.getAttribute(CLIENT_ID_ATTRIBUTE))
                .map(Object::toString)
                .orElse("-");
    }

    public static String getStatus(HttpServletRequest req, HttpServletResponse resp) {
        if ("true".equals(req.getAttribute(CLIENT_DISCONNECTED))) {
            return "499";
        }
        return String.valueOf(resp.getStatus());
    }

    public static String makeHeadersLogLine(Tuple2List<String, String> headers) {
        return new JsonObject(new Tuple2List<>(headers.map(s -> Tuple2.tuple(s.get1(), calculateValue(s))))).serialize();
    }

    private static JsonValue calculateValue(Tuple2<String, String> t) {
        return JsonString.valueOf(HttpRequestUtils.encodeSensitiveHeader(t)._2);
    }

    public static Function0<DiskLog4jRequestLog> consF() {
        return DiskLog4jRequestLog::new;
    }
}
