package ru.yandex.chemodan.log;

import java.io.Serializable;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.AsyncAppender;
import org.apache.logging.log4j.core.appender.ConsoleAppender;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.apache.logging.log4j.core.appender.RollingRandomAccessFileAppender;
import org.apache.logging.log4j.core.async.AsyncLoggerConfig;
import org.apache.logging.log4j.core.config.AppenderRef;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.Configurator;
import org.apache.logging.log4j.core.config.DefaultConfiguration;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.apache.logging.log4j.core.layout.PatternLayout;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.log.utils.HupTriggeringPolicy;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.log4j2.mlf.Log4j2Logger;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.FieldX;
import ru.yandex.misc.version.AppName;

/**
 * @author akirakozov
 * @author vpronto
 */
public class Log4jHelper {

    private static final int ASYNC_APPENDER_BUFFER = 16 * 1024;
    private static final int DEF_NIO_BUFF_SIZE = 500 * 1024 * 1024;
    private static final int LOW_LIMIT_NIO_RND_BUFFER_SIZE = 8 * 1024;
    public static final String accessLogName = "ru.yandex.misc.web.servletContainer.jetty.access";
    public static final String accessLogTskvName = "ru.yandex.misc.web.servletContainer.jetty.tskv.access";
    public static final String monicaLogName = "ru.yandex.misc.monica";
    public static final String apacheRetryExecLogName = "org.apache.http.impl.execchain.RetryExec";
    private static final ru.yandex.misc.log.mlf.Level DEF_MISC_LEVEL = ru.yandex.misc.log.mlf.Level.DEBUG;
    private static final ru.yandex.misc.log.mlf.Level DEF_ROOT_LEVEL = ru.yandex.misc.log.mlf.Level.INFO;
    public static final String ROOT = LogManager.ROOT_LOGGER_NAME;

    static {
        System.setProperty("org.eclipse.jetty.util.log.class", "ru.yandex.chemodan.log.Jetty2MlfBridge");
        System.setProperty("log4j.configurationFile", "log4j2-chemodan.xml");
        org.eclipse.jetty.util.log.Log.setLog(new Jetty2MlfBridge(ROOT));
        org.eclipse.jetty.util.log.Log.getRootLogger().setDebugEnabled(false);
        new org.eclipse.jetty.util.log.JavaUtilLog().setDebugEnabled(false);
    }

    public static void configureLogger(Class claz, ru.yandex.misc.log.mlf.Level level) {
        configureLogger(claz.getName(), level);
    }

    public static void configureLogger(String loggerName, ru.yandex.misc.log.mlf.Level level) {
        configureLogger(loggerName, Log4j2Logger.getLevel(level), true);
    }

    public static void configureNonAdditiveLogger(String loggerName, Level level) {
        configureLogger(loggerName, level, false);
    }

    private static void configureLogger(String loggerName, Level level, boolean additive) {
        Logger logger = LogManager.getLogger(loggerName);
        Configuration configuration = getContext().getConfiguration();
        LoggerConfig loggerConfig = new LoggerConfig(
                loggerName,
                level,
                additive);
        FieldX includeLocation = ClassX.wrap(LoggerConfig.class).getDeclaredField("includeLocation");
        includeLocation.setAccessibleTrueReturnThis();
        includeLocation.set(loggerConfig, false);
        configuration.addLogger(loggerName, loggerConfig);
        Configurator.setLevel(logger.getName(), level);
    }

    private static void configureRootLogger(
            AppName appName,
            ru.yandex.misc.log.mlf.Level rootLevel,
            ru.yandex.misc.log.mlf.Level miscLevel,
            int bufferSize,
            boolean async,
            Layout<? extends Serializable> layout,
            String postfix,
            boolean measureExceptions)
    {
        initLogger(rootLevel, miscLevel);
        if (measureExceptions) {
            initExceptionCountingLogger();
        }

        ReopenOnHupAppenderBuilder builder = appenderBuilder()
                .appName(appName)
                .async(async)
                .level(rootLevel)
                .buffSize(bufferSize)
                .postfix(postfix);

        if (layout != null) {
            builder.layout(layout);
        } else {
            builder.layout(new TskvLogPatternLayout(
                            "ydisk-" + appName.appName() + "-log", false));

        }
        builder.build();
    }

    private static void configureAccessLogger(
            AppName appName, int bufferSize, boolean tskvOnly, Option<String> tskvFormat,
            Option<Layout<? extends Serializable>> layout, ru.yandex.misc.log.mlf.Level level)
    {
        configureAccessLoggerBase(appName, bufferSize, true, tskvFormat, layout, level, accessLogTskvName);

        if (tskvOnly) {
            //deactivate common access log
            configureLogger(accessLogName, ru.yandex.misc.log.mlf.Level.ERROR);
        } else {
            configureAccessLoggerBase(appName, bufferSize, false, Option.empty(), layout, level, accessLogName);
        }
    }

    private static void configureAccessLoggerBase(
            AppName appName, int bufferSize, boolean tskv, Option<String> tskvFormat,
            Option<Layout<? extends Serializable>> layoutO, ru.yandex.misc.log.mlf.Level level, String logName)
    {
        Layout<? extends Serializable> layout = layoutO.getOrElse(new PatternLayoutBridge(tskv
                ? getTskvLogPatternLayout(tskvFormat.getOrElse("ydisk-java-access-log-" + appName.appName()))
                : new CommonAccessLogPatternLayout()));

        appenderBuilder()
                .appName(appName)
                .postfix("-access")
                .tskvLog(tskv)
                .layout(layout)
                .level(level)
                .name(logName)
                .buffSize(bufferSize)
                .build();

    }

    protected static TskvLogPatternLayout getTskvLogPatternLayout(String tskvFormat) {
        TskvLogPatternLayout tskvLogPatternLayout = new TskvLogPatternLayout(tskvFormat, false, true);
        tskvLogPatternLayout.setForceEmptyRidAndYcrid(tskvFormat.contains("access-log"));
        return tskvLogPatternLayout;
    }

    public static void configureHttpClientLoggerLevel(ru.yandex.misc.log.mlf.Level level) {
        configureLogger("ru.yandex.misc.io.http.apache.v4", level);

    }

    public static void configureAliveAppsLoggerLevel(ru.yandex.misc.log.mlf.Level level) {
        configureLogger("ru.yandex.commune.alive2", level);
    }

    private static String createLogFileName(AppName appName, String postfix, boolean isTskvLog) {
        String logFileName = appName.appName() + postfix + (isTskvLog ? "-tskv" : "");
        return "/var/log/yandex/" + appName.serviceName() + "/" + logFileName + ".log";
    }

    //please prefer to use ru.yandex.chemodan.log.Log4jHelper.rootLoggerBuilder
    public static void initLogger(ru.yandex.misc.log.mlf.Level rootLevel, ru.yandex.misc.log.mlf.Level miscLevel) {

        initRootLogger(rootLevel);
        initMiscLoggers(miscLevel);
        initMonicaLogger();
        initRetryExecLogger();
    }

    public static void initRootLogger(ru.yandex.misc.log.mlf.Level level) {

        LoggerContext ctx = getContext();
        Configuration config = ctx.getConfiguration();

        LoggerConfig rootLogger = config.getRootLogger();
        rootLogger.setAdditive(false);
        rootLogger.getAppenders().keySet().forEach(a -> config.getRootLogger().removeAppender(a));
        rootLogger.setLevel(Log4j2Logger.getLevel(level));
        ctx.updateLoggers();
        ctx.reconfigure();
    }

    private static void initMiscLoggers(ru.yandex.misc.log.mlf.Level level) {
        configureLogger("ru.yandex", level);
    }

    private static void initMonicaLogger() {
        configureLogger(monicaLogName, ru.yandex.misc.log.mlf.Level.WARN);
    }

    private static void initRetryExecLogger() {
        configureLogger(apacheRetryExecLogName, ru.yandex.misc.log.mlf.Level.WARN);
    }

    private static void initExceptionCountingLogger() {
        MonicaMeasureExceptionsAppender appender = new MonicaMeasureExceptionsAppender();
        addAppender(appender, appender.getName(), ru.yandex.misc.log.mlf.Level.INFO, true, true);
    }

    private static void consReopenOnHupAppender(
            AppName appName,
            String postfix,
            boolean isTskvLog,
            Layout<? extends Serializable> layout,
            ru.yandex.misc.log.mlf.Level level,
            String name,
            String fileName,
            boolean async,
            int buffSize)
    {
        final LoggerContext ctx = getContext();
        Configuration config = ctx.getConfiguration();

        Appender appender;
        if (buffSize >= LOW_LIMIT_NIO_RND_BUFFER_SIZE) {
            appender = RollingRandomAccessFileAppender.newBuilder()
                    .withFileName(fileName != null ? fileName : createLogFileName(appName, postfix, isTskvLog))
                    .withAppend(true)
                    .withStrategy(new HupRolloverStrategy())
                    .withPolicy(new HupTriggeringPolicy())
                    .withFilePattern("")
                    .withName(name)
                    .withImmediateFlush(false)
                    .withBufferSize(buffSize)
                    .withLayout(layout)
                    .setConfiguration(config)
                    .build();
        } else {
            appender = RollingFileAppender.newBuilder()
                    .withFileName(fileName != null ? fileName : createLogFileName(appName, postfix, isTskvLog))
                    .withAppend(true)
                    .withIgnoreExceptions(false)
                    .withStrategy(new HupRolloverStrategy())
                    .withPolicy(new HupTriggeringPolicy())
                    .withFilePattern("")
                    .withName(name)
                    .withImmediateFlush(!async)
                    .withBufferedIo(buffSize > 0)
                    .withBufferSize(buffSize)
                    .withLayout(layout)
                    .setConfiguration(config)
                    .build();
        }

        addAppender(appender, name, level, async, false);
    }

    public static void addAppender(
            Appender appender,
            String name,
            ru.yandex.misc.log.mlf.Level level,
            boolean async, boolean appendToRoot)
    {
        final LoggerContext ctx = getContext();
        Configuration config = ctx.getConfiguration();
        appender.start();
        config.addAppender(appender);
        AppenderRef[] refs = new AppenderRef[]{AppenderRef.createAppenderRef(appender.getName(), null, null)};
        /**
         * TODO: remove it! should be enough log4j2.contextSelector to org.apache.logging.log4j.core.async.AsyncLoggerContextSelector
         * (but in test this configureation was faster)
         */
        if (async) {
            appender = AsyncAppender.newBuilder()
                    .setConfiguration(config)
                    .setAppenderRefs(refs)
                    .setName(appender.getName())
                    .setIncludeLocation(false)
                    .setBufferSize(ASYNC_APPENDER_BUFFER)
                    .build();
            appender.start();
        }
        if (ROOT.equals(name) || appendToRoot) {
            config.getRootLogger().addAppender(appender, null, null);
            return;
        }
        LoggerConfig loggerConfig = async ?
                AsyncLoggerConfig.createLogger(
                        false,
                        Log4j2Logger.getLevel(level),
                        name,
                        "false",
                        refs,
                        null,
                        config,
                        null) :
                LoggerConfig.createLogger(
                        false,
                        Log4j2Logger.getLevel(level),
                        name,
                        "false",
                        refs,
                        null,
                        config,
                        null);

        loggerConfig.addAppender(appender, Log4j2Logger.getLevel(level), null);
        config.addLogger(name, loggerConfig);
        config.start();
        ctx.updateLoggers();

    }

    public static void configureTestLogger() {
        configureTestLogger(ru.yandex.misc.log.mlf.Level.DEBUG);
    }

    public static void configureTestLogger(ru.yandex.misc.log.mlf.Level level) {
        System.setProperty("log4j.configurationFile", "log4j2-test.xml");
        configureConsoleLogger(level);
    }

    public static void configureConsoleLogger(ru.yandex.misc.log.mlf.Level level) {

        LoggerContext ctx = getContext();
        Configuration config = ctx.getConfiguration();
        LoggerConfig rootLogger = config.getRootLogger();

        Level lev = Log4j2Logger.getLevel(level);
        ConsoleAppender stdout = consConsoleAppender(config);
        stdout.start();

        rootLogger.addAppender(stdout, lev, null);
        ctx.updateLoggers();
    }

    public static ConsoleAppender consConsoleAppender() {
        return consConsoleAppender(getContext().getConfiguration());
    }

    private static ConsoleAppender consConsoleAppender(Configuration config) {
        return ConsoleAppender.createDefaultAppenderForLayout(
                PatternLayout.newBuilder()
                        .withPattern(DefaultConfiguration.DEFAULT_PATTERN)
                        .withConfiguration(config)
                        .build()
        );
    }

    public static class RootLoggerBuilder {
        private AppName appName;
        private boolean async = true;
        private ru.yandex.misc.log.mlf.Level miscLevel = DEF_MISC_LEVEL;
        private ru.yandex.misc.log.mlf.Level rootLevel = DEF_ROOT_LEVEL;
        private int buffSize = DEF_NIO_BUFF_SIZE;
        private Layout<? extends Serializable> layout;
        private String postfix = "";
        private boolean measureExceptions = true;

        public RootLoggerBuilder appName(AppName appName) {
            this.appName = appName;
            return this;
        }

        public RootLoggerBuilder rootLevel(ru.yandex.misc.log.mlf.Level rootLevel) {
            this.rootLevel = rootLevel;
            return this;
        }

        public RootLoggerBuilder miscLevel(ru.yandex.misc.log.mlf.Level miscLevels) {
            this.miscLevel = miscLevels;
            return this;
        }

        public RootLoggerBuilder async(boolean async) {
            this.async = async;
            return this;
        }

        public RootLoggerBuilder bufferSize(int buffSize) {
            this.buffSize = buffSize;
            return this;
        }

        public RootLoggerBuilder layout(Layout<? extends Serializable> layout) {
            this.layout = layout;
            return this;
        }

        public RootLoggerBuilder postfix(String postfix) {
            this.postfix = postfix;
            return this;
        }

        public RootLoggerBuilder measureExceptions(boolean measureExceptions) {
            this.measureExceptions = measureExceptions;
            return this;
        }

        public void build() {
            Validate.notNull(appName, "appName can't be null");
            configureRootLogger(appName, rootLevel, miscLevel, buffSize, async, layout, postfix, measureExceptions);
        }
    }

    public static RootLoggerBuilder rootLoggerBuilder() {
        RootLoggerBuilder builder = new RootLoggerBuilder();
        String miscLevel = System.getProperty("root.logger.misc.level");
        if (miscLevel != null) {
            builder = builder.miscLevel(ru.yandex.misc.log.mlf.Level.valueOf(miscLevel));
        }

        String rootLevel = System.getProperty("root.logger.root.level");
        if (rootLevel != null) {
            builder = builder.rootLevel(ru.yandex.misc.log.mlf.Level.valueOf(rootLevel));
        }

        return builder;
    }

    public static class ReopenOnHupAppenderBuilder {
        private AppName appName;
        private String fileName;
        private String postfix = "";
        private boolean isTskvLog = true;
        private Layout<? extends Serializable> layout = PatternLayout.createDefaultLayout();
        private ru.yandex.misc.log.mlf.Level level = DEF_MISC_LEVEL;
        private String name = ROOT;
        private boolean async = true;
        private int buffSize = DEF_NIO_BUFF_SIZE;

        public ReopenOnHupAppenderBuilder appName(AppName appName) {
            this.appName = appName;
            return this;
        }

        public ReopenOnHupAppenderBuilder postfix(String postfix) {
            this.postfix = postfix;
            return this;
        }

        public ReopenOnHupAppenderBuilder tskvLog(boolean tskvLog) {
            isTskvLog = tskvLog;
            return this;
        }

        public ReopenOnHupAppenderBuilder layout(Layout<? extends Serializable> layout) {
            this.layout = layout;
            return this;
        }

        @Deprecated
        public ReopenOnHupAppenderBuilder layout(org.apache.log4j.PatternLayout layout) {
            this.layout = wrapLegacyLayout(layout);
            return this;
        }

        public ReopenOnHupAppenderBuilder level(ru.yandex.misc.log.mlf.Level level) {
            this.level = level;
            return this;
        }

        public ReopenOnHupAppenderBuilder name(String name) {
            this.name = name;
            return this;
        }

        /**
         * size of nio buffer, if size less then @see LOW_LIMIT_NIO_RND_BUFFER_SIZE
         * RollingFileAppender will be used instead of fast RollingRandomAccessFileAppender
         */
        public ReopenOnHupAppenderBuilder buffSize(int buffSize) {
            this.buffSize = buffSize;
            return this;
        }

        public ReopenOnHupAppenderBuilder fileName(String fileName) {
            this.fileName = fileName;
            return this;
        }

        public ReopenOnHupAppenderBuilder async(boolean async) {
            this.async = async;
            return this;
        }

        public void build() {
            Validate.notNull(appName, "appName can't be null");
            consReopenOnHupAppender(appName,
                    postfix,
                    isTskvLog,
                    layout,
                    level,
                    name,
                    fileName,
                    async,
                    buffSize);
        }
    }

    public static ReopenOnHupAppenderBuilder appenderBuilder() {
        return new ReopenOnHupAppenderBuilder();
    }

    public static class AccessLoggerAppenderBuilder {
        private AppName appName;
        private int bufferSize = DEF_NIO_BUFF_SIZE;
        private boolean tskvOnly = true;
        private Option<String> tskvFormat = Option.empty();
        private Option<Layout<? extends Serializable>> layout = Option.empty();
        private ru.yandex.misc.log.mlf.Level level = DEF_MISC_LEVEL;

        public AccessLoggerAppenderBuilder appName(AppName appName) {
            this.appName = appName;
            return this;
        }

        public AccessLoggerAppenderBuilder bufferSize(int bufferSize) {
            this.bufferSize = bufferSize;
            return this;
        }

        public AccessLoggerAppenderBuilder tskvOnly(boolean tskvOnly) {
            this.tskvOnly = tskvOnly;
            return this;
        }

        public AccessLoggerAppenderBuilder tskvFormat(String tskvFormat) {
            this.tskvFormat = Option.of(tskvFormat);
            return this;
        }

        public AccessLoggerAppenderBuilder layout(Layout<? extends Serializable> layout) {
            this.layout = Option.of(layout);
            return this;
        }

        public AccessLoggerAppenderBuilder level(ru.yandex.misc.log.mlf.Level level) {
            this.level = level;
            return this;
        }

        public void build() {
            configureAccessLogger(appName, bufferSize, tskvOnly, tskvFormat, layout, level);
        }
    }

    public static AccessLoggerAppenderBuilder accessLoggerBuilder() {
        return new AccessLoggerAppenderBuilder();
    }

    @Deprecated
    public static Layout<? extends Serializable> wrapLegacyLayout(org.apache.log4j.PatternLayout layout) {
        return new PatternLayoutBridge(layout);
    }

    private static LoggerContext getContext() {
        return (LoggerContext) LogManager.getContext(false);
    }
}
