package ru.yandex.chemodan.log;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.log.utils.CurryingLogger;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.mlf.format.Slf4jLikeMessageFormatter;
import ru.yandex.misc.log.mlf.ndc.Ndc;

/**
 * @author dbrylev
 */
public class LoggerProxies {

    public static Logger withYcrid(Logger logger, Option<String> ycrid) {
        return ycrid.map(rid -> LoggerProxies.withNdc(logger, "#ycrid=" + rid)).getOrElse(logger);
    }

    public static Logger withNdc(Logger logger, String message) {
        return proxy((proxy, method, args) -> {
            Ndc.Handle handle = Ndc.push(message);
            try {
                return method.invoke(logger, args);

            } finally {
                handle.popSafely();
            }
        });
    }

    public static Logger noStackTrace(Logger logger) {
        return new CurryingLogger() {
            @Override
            public boolean isEnabledFor(Level level) {
                return logger.isEnabledFor(level);
            }

            @Override
            public void log(Level level, Object message) {
                logger.log(level, message);
            }

            @Override
            public void log(Level level, String message) {
                logger.log(level, message);
            }

            @Override
            public void log(Level level, Throwable t) {
                logger.log(level, ExceptionUtils.getAllMessages(t));
            }

            @Override
            public void log(Level level, Object message, Throwable t) {
                logger.log(level, "{}: {}", message, ExceptionUtils.getAllMessages(t));
            }

            @Override
            public void log(Level level, String message, Throwable t) {
                log(level, message, (Object) t);
            }

            @Override
            public void log(Level level, String message, Object... args) {
                Object last = args.length > 0 ? args[args.length - 1] : null;

                if (last instanceof Throwable) {
                    args[args.length - 1] = ExceptionUtils.getAllMessages((Throwable) last);

                    if (countPlaceholders(message) >= args.length) {
                        logger.log(level, message, args);
                    } else {
                        logger.log(level, message + ": {}", args);
                    }
                } else {
                    logger.log(level, message, args);
                }
            }
        };
    }

    public static Logger noStackTrace(Class<?> clazz) {
        return noStackTrace(LoggerFactory.getLogger(clazz));
    }

    public static Logger proxy(InvocationHandler handler) {
        return (Logger) Proxy.newProxyInstance(
                LoggerProxies.class.getClassLoader(), new Class<?>[] { Logger.class }, handler);
    }

    static int countPlaceholders(String message) {
        if (message == null) {
            return 0;
        }

        int count = 0, index;

        for (int from = 0; from < message.length(); from = index + Slf4jLikeMessageFormatter.PLACEHOLDER.length()) {
            index = message.indexOf(Slf4jLikeMessageFormatter.PLACEHOLDER, from);

            if (index == -1) {
                break;
            }
            if (index == 0
                || message.charAt(index - 1) != Slf4jLikeMessageFormatter.ESCAPE_CHAR
                || index > 1 && message.charAt(index - 2) == Slf4jLikeMessageFormatter.ESCAPE_CHAR)
            {
                ++count;
            }
        }
        return count;
    }
}
