package ru.yandex.chemodan.log;

import java.util.LinkedHashMap;
import java.util.concurrent.Callable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.ndc.Ndc;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class TskvNdcUtil {
    private static final Pattern NDC_FIELD_PATTERN = Pattern.compile("#([^=]+)=([^#]+)(?=,#|$)");

    public static Ndc.Handle pushToNdc(String key, Object value) {
        return pushToNdc(Tuple2List.fromPairs(key, value));
    }

    public static Ndc.Handle pushToNdc(Tuple2List<String, Object> values) {
        String prefix = StringUtils.isNotBlank(Ndc.get()) ? "#" : "";
        String ndc = StringUtils.join(values.map((key, value) -> key + "=" + value), ",#");

        return Ndc.push(prefix + ndc);
    }

    public static void runWithNdc(Runnable runnable, Object... pairs) {
        runWithNdc(runnable, Tuple2List.fromPairs(pairs).map1(Object::toString));
    }

    public static void runWithNdc(Runnable runnable, Tuple2List<String, Object> values) {
        if (values.isNotEmpty()) {
            Ndc.Handle handle = pushToNdc(values);
            try {
                runnable.run();
            } finally {
                handle.popSafely();
            }
        } else {
            runnable.run();
        }
    }

    public static void runWithTemporaryNdcReplace(String ndc, Runnable runnable) {
        execWithTemporaryNdcReplace(ndc, () -> {
            runnable.run();
            return null;
        });
    }

    public static <V> V execWithTemporaryNdcReplace(String ndc, Callable<V> callable) {
        Ndc.Handle prevHandle = new Ndc.Handle(Ndc.get());
        new Ndc.Handle(ndc).popSafely();
        try {
            return callable.call();
        } catch(Exception e) {
            throw ExceptionUtils.translate(e);
        } finally {
            prevHandle.popSafely();
        }
    }

    public static MapF<String, String> parseNdc(String ndc) {
        if (StringUtils.isBlank(ndc)) {
            return Cf.map();
        }

        MapF<String, String> result = Cf.wrap(new LinkedHashMap<String, String>());
        Matcher matcher = NDC_FIELD_PATTERN.matcher(ndc);
        while (matcher.find()) {
            addToNdcMap(matcher.group(1), matcher.group(2), result);
        }
        return result;
    }

    private static void addToNdcMap(String key, String value, MapF<String, String> ndc) {
        if (key.endsWith("+")) {
            key = key.substring(0, key.length() - 1);
            ndc.put(key, ndc.getOrElse(key, "") + value);
        } else {
            ndc.put(key, value);
        }
    }
}
