package ru.yandex.chemodan.mime;

import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.exec.ShellUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author bursy
 */
@RequiredArgsConstructor
public class LibMagicMimeTypeDetector {
    private static final ListF<String> ARGS = Cf.list("--mime-type", "-b");

    private final String fileCommandPath;
    private final ListF<String> additionalArgs;
    private final Duration timeout;
    private final ListF<MimeTypeCorrector> correctors;

    // https://stackoverflow.com/questions/43473056/which-mime-type-should-be-used-for-a-raw-image
    private static final MapF<String, String> RAW_IMAGE_EXTENSION_OVERRIDES = Tuple2List.<String, String>fromPairs(
            ".arw", "image/x-sony-arw",
            ".cr2", "image/x-canon-cr2",
            ".crw", "image/x-canon-crw",
            ".dcr", "image/x-kodak-dcr",
            ".dng", "image/x-adobe-dng",
            ".erf", "image/x-epson-erf",
            ".k25", "image/x-kodak-k25",
            ".kdc", "image/x-kodak-kdc",
            ".mrw", "image/x-minolta-mrw",
            ".nef", "image/x-nikon-nef",
            ".orf", "image/x-olympus-orf",
            ".pef", "image/x-pentax-pef",
            ".raf", "image/x-fuji-raf",
            ".raw", "image/x-panasonic-raw",
            ".sr2", "image/x-sony-sr2",
            ".srf", "image/x-sony-srf",
            ".x3f", "image/x-sigma-x3f"
    ).toMap();

    private static final MapF<String, String> ZIP_EXTENSION_OVERRIDES = Tuple2List.<String, String>fromPairs(
            ".apk", "application/vnd.android.package-archive",
            ".docx", "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
            ".xlsx", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
    ).toMap();

    private static final MapF<String, String> XML_EXTENSION_OVERRIDES = Tuple2List.<String, String>fromPairs(
            ".fb2", "application/x-fictionbook+xml"
    ).toMap();

    // extension overrides are mapped to originally detected mimetype
    // this is used to provide better precision for specific types, not to ignore libmagic results
    private static final MapF<String, MapF<String, String>> ALL_EXTENSION_OVERRIDES = Tuple2List.fromPairs(
            "image/tiff", RAW_IMAGE_EXTENSION_OVERRIDES,
            "application/zip", ZIP_EXTENSION_OVERRIDES,
            "text/xml", XML_EXTENSION_OVERRIDES
    ).toMap();

    public Option<String> detectWithoutOverrides(File2 file) {
        String script = createScript(file);

        ExecResult result = ShellUtils.executeGrabbingOutput(script, (int) timeout.getMillis());

        return Option.ofNullable(result.getStdout())
                .map(String::trim)
                .filter(StringUtils::isNotEmpty);
    }

    public Option<String> detectWithOverrides(File2 file, Option<String> filename) {
        return detectWithoutOverrides(file)
                .map(mimeType -> applyExtensionOverrides(mimeType, filename))
                .map(mimeType -> applyCorrections(mimeType, file, filename));
    }

    private String applyExtensionOverrides(String mimeType, Option<String> filenameO) {
        return filenameO.map(String::toLowerCase).filterMap(filename ->
                ALL_EXTENSION_OVERRIDES.getO(mimeType)
                        .filterMap(overrides -> overrides
                                .filterKeys(filename::endsWith)
                                .values().iterator().nextO()))
                .getOrElse(mimeType);
    }

    private String applyCorrections(String mimeType, File2 file, Option<String> filename) {
        return correctors.filter(corrector -> corrector.isApplicable(mimeType))
                .firstO()
                .filterMap(corrector -> corrector.correct(file, filename))
                .getOrElse(mimeType);
    }

    private String createScript(File2 file) {
        ListF<String> cmd = Cf.arrayList(fileCommandPath);
        cmd.addAll(ARGS);
        cmd.addAll(additionalArgs);
        cmd.add(file.getAbsolutePath());

        return ShellUtils.commandToScript(cmd);
    }
}
