package ru.yandex.chemodan.monica.intervaldistribution;

import org.joda.time.Duration;
import org.joda.time.ReadableDuration;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.impl.test.Generator;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.monica.core.blocks.DistributionData;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public abstract class AbstractComparableDistributionTest<T extends AbstractComparableIntervalGauge> {
    protected abstract DistributionData extractDistributionData(T distribution);

    protected <C extends Comparable<C>> T createGauge(ListF<C> bounds) {
        return createGauge(bounds, C::toString);
    }

    protected abstract <C extends Comparable<C>> T createGauge(ListF<C> bounds, Function<C, String> boundNameF);

    @Test
    public void testForDuration() {
        T gaugeT = createGauge(
                Cf.list(
                        Duration.standardSeconds(40),
                        Duration.standardMinutes(20),
                        Duration.standardHours(12)
                ),
                duration -> TimeUtils.formatSimple(duration.toPeriod()));

        @SuppressWarnings("unchecked")
        AbstractComparableIntervalGauge<ReadableDuration> gauge =
                (AbstractComparableIntervalGauge<ReadableDuration>)gaugeT;

        Generator.ints(0, 41).take(23).forEachRemaining(secs -> gauge.inc(Duration.standardSeconds(secs)));
        Generator.ints(1, 21).take(39).forEachRemaining(mins -> gauge.inc(Duration.standardMinutes(mins)));
        Generator.ints(1, 13).take(11).forEachRemaining(hours -> gauge.inc(Duration.standardHours(hours)));
        Generator.ints(13, 49).take(27).forEachRemaining(hours -> gauge.inc(Duration.standardHours(hours)));

        DistributionData data = extractDistributionData(gaugeT);

        Assert.equals(23.0, data.fractions.getTs("40sec"), 0.01);
        Assert.equals(39.0, data.fractions.getTs("20min"), 0.01);
        Assert.equals(11.0, data.fractions.getTs("12hours"), 0.01);
        Assert.equals(27.0, data.fractions.getTs("other"), 0.01);
//        Assert.equals(100.0, data.summary);
    }

    @Test
    public void testForLong() {
        T gaugeT = createGauge(Cf.list(39L, 10L, 5L, 2L));

        @SuppressWarnings("unchecked")
        AbstractComparableIntervalGauge<Long> distribution = (AbstractComparableIntervalGauge<Long>) gaugeT;

        // -Infinity < N <= 2
        distribution.inc(-123456L);
        distribution.inc(2L);

        // 2 < N <= 5
        distribution.inc(3L);
        distribution.inc(3L);
        distribution.inc(4L);
        distribution.inc(5L);

        // 5 < N <= 10
        distribution.inc(9L);

        // N > 39 (other)
        distribution.inc(40L);
        distribution.inc(12345L);
        distribution.inc(2452345L);

        DistributionData data = extractDistributionData(gaugeT);

        Assert.equals(20.0, data.fractions.getTs("2"), 0.01);
        Assert.equals(40.0, data.fractions.getTs("5"), 0.01);
        Assert.equals(10.0, data.fractions.getTs("10"), 0.01);
        Assert.equals(30.0, data.fractions.getTs("other"), 0.01);
//        Assert.equals(10.0, data.summary);
    }
}
