package ru.yandex.chemodan.mpfs;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Optional;

import org.apache.http.HttpRequestInterceptor;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author dbrylev
 */
@Configuration
public class MpfsClientContextConfiguration {

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    public @interface Mpfs {
    }

    @Bean
    public MpfsClient mpfsClient(
            @Value("${mpfs.host}") String host,
            @Value("${mpfs.init-on-demand}") boolean initOnDemand,
            Optional<MpfsUserInitParamsExtractor> userInitParamsExtractor,
            @Mpfs Optional<HttpRequestInterceptor> customRequestInterceptor,
            @Value("${djfs.host:-}") String djfsHost,
            @Qualifier("djfs")
            Optional<HttpClientConfigurator> djfsHttpClientConfigurator,
            @Value("${mpfs.ignore-user-not-initialized-exception}") boolean ignoreUserNotInitializedException)
    {
        ApacheHttpClientUtils.Builder builder = mpfsHttpClientConfigurator().createBuilder();
        ApacheHttpClientUtils.Builder longBuilder = mpfsLongHttpClientConfigurator().createBuilder();

        if (customRequestInterceptor.isPresent()) {
            builder.withInterceptorLast(customRequestInterceptor.get());
            longBuilder.withInterceptorLast(customRequestInterceptor.get());
        }

        return new MpfsClientImpl(builder.build(), longBuilder.build(),
                host, userInitParamsExtractor.orElseGet(MpfsUserInitParamsExtractor::empty), initOnDemand,
                StringUtils.isBlank(djfsHost) ?
                        Option.empty() :
                        Option.of(djfsHost),
                Option.x(djfsHttpClientConfigurator.map(HttpClientConfigurator::configure)), ignoreUserNotInitializedException);
    }

    @Bean
    @Mpfs
    @OverridableValuePrefix("mpfs")
    public HttpClientConfigurator mpfsHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    @Mpfs
    @OverridableValuePrefix("mpfs_long")
    public HttpClientConfigurator mpfsLongHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    @OverridableValuePrefix("djfs")
    @Qualifier("djfs")
    public HttpClientConfigurator djfsHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }
}
