package ru.yandex.chemodan.mpfs;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ByteArrayEntity;
import org.glassfish.grizzly.http.util.Header;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockFullDescription;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockItemDescription;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.commune.json.serialize.JsonSerializer;
import ru.yandex.commune.json.write.JacksonJsonWriterFactory;
import ru.yandex.commune.json.write.JsonWriter;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.parse.JacksonJsonNodeWrapper;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadStringResponseHandler;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


/**
 * http://wiki.yandex-team.ru/pochta/ya.disk/MPFS/api/Kladun
 *
 * @author akirakozov
 * @author ssytnik
 */
public class MpfsClientImpl implements MpfsClient {

    private static final Logger logger = LoggerFactory.getLogger(MpfsClientImpl.class);

    private static final List<Integer> suppressedCodes = Collections.singletonList(404);

    private static final String META_TAGS = "file_mid,size,mimetype,hid,mediatype";
    private static final String RESOURCE_INFO_META_TAGS = "uid,mediatype,short_url,preview";
    private static final String XIVA_LIMIT_EXCEEDED = "254";
    private static final Timeout DEFAULT_TIMEOUT = Timeout.timeout(5, TimeUnit.SECONDS);
    private static final int DEFAULT_MAX_CONNECTIONS = 300;

    public static final String DISK_PATH_HEADER = "X-Disk-Path";

    private static final ObjectMapper objectMapper = new ObjectMapper();

    private static final String BULK_INFO_BY_RESOURCE_IDS_ENDPOINT = "json/bulk_info_by_resource_ids";
    private static final String BULK_INFO_ENDPOINT = "json/bulk_info";
    private static final String INFO_ENDPOINT = "json/info";
    private static final String DEFAULT_FOLDERS_ENDPOINT = "json/default_folders";

    private final MpfsRequestExecutor executor;
    private final MpfsUserInitParamsExtractor userInitParams;
    private String mpfsHost;
    private final Option<String> djfsHost;
    private final Option<HttpClient> djfsHttpClient;
    private final boolean ignoreUserNotInitializedException;

    private final DynamicProperty<ListF<String>> infoSupportedMetaFields = new DynamicProperty<>("djfs-client-info" +
            "-supported-meta-fields",
            Cf.list(
                    "blocked", "download_counter", "views_counter", "drweb", "fotki_data_url", "from", "group",
                    "mediatype",
                    "mimetype", "original_parent_id", "public", "search_scope", "shared_rights", "short_url", "size",
                    "storage_type", "sizes", "with_shared", "append_time", "photoslice_time", "file_id",
                    "office_online_url",
                    "total_results_count", "resource_id", "versioning_status", "video_info", "albums_exclusions",
                    "version|urn:yandex:disk:dist", "cdn|urn:yandex:disk:dist:url", "sha256", "md5", "file_url",
                    "aesthetics", "height", "photoslice_album_type", "custom_properties", "etime", "width",
                    "autouploaded",
                    "media_type", "revision", "public_hash", "angle", "custom_preview", "comment_ids", "note_name",
                    "note_revision_created", "note_revision_deleted", "file_mid", "digest_mid",
                    "pmid", "link|urn:yandex:disk:client:meta", "empty", "visible", "Win32FileAttributes|urn:schemas" +
                            "-microsoft-com:",
                    "fs_symbolic_link", "fotki_image_id", "hid"
            ));

    private final DynamicProperty<ListF<String>> infoSupportedParams = new DynamicProperty<>("djfs-client-info" +
            "-supported-params",
            Cf.list(
                    "uid", "path", "preview_size", "preview_crop", "preview_quality",
                    "preview_allow_big_size", "preview_type", "preview_animate", "tld"
            ));

    private final DynamicProperty<ListF<String>> bulkInfoSupportedParams = new DynamicProperty<>("djfs-client" +
            "-bulk_info-supported-params",
            Cf.list(
                    "uid", "preview_size", "preview_crop", "preview_quality", "preview_allow_big_size"
            ));

    private final DynamicProperty<ListF<String>> bulkInfoSupportedMetaFields = new DynamicProperty<>("djfs-client" +
            "-bulk_info-supported-meta-fields",
            Cf.list(
                    "md5", "mimetype", "fs_symbolic_link", "visible", "empty", "group", "file_id", "short_url", "angle",
                    "mediatype", "photoslice_album_type", "width", "albums_exclusions", "video_info", "aesthetics",
                    "pmid", "height", "alias_enabled", "resource_id", "sha256", "hasfolders", "numfolders", "numfiles",
                    "drweb", "etime", "size", "sizes", "custom_preview", "file_url", "media_type", "public", "file_mid",
                    "storage_type", "photoslice_time", "versioning_status", "fotki_data_stid", "fotki_data_url",
                    "note_name", "note_revision_created", "note_revision_deleted", "blocked", "custom_properties",
                    "download_counter", "views_counter", "page_blocked_items_num", "comment_ids", "blockings",
                    "revision",
                    "public_hash", "append_time", "original_id", "folder_type", "numchildren", "autouploaded",
                    "digest_mid"
            ));

    private final DynamicProperty<ListF<String>> bulkInfoByResorceIdsSupportedParams = new DynamicProperty<>("djfs" +
            "-client-bulk_info_by_resource_ids-supported-params",
            Cf.list(
                    "uid", "enable_service_ids", "preview_size", "preview_crop", "preview_quality",
                    "preview_allow_big_size"
            ));

    private final DynamicProperty<ListF<String>> djfsSupportedRootFolders = new DynamicProperty<>("djfs-client" +
            "-supported-root-paths", Cf.list(
            "/disk", "/trash", "/photounlim", "/hidden", "/attach", "/lnarod", "/misc", "/notes", "/additional",
            "/share"
    ));

    private final DynamicProperty<ListF<String>> defaultFoldersSupportedParams = new DynamicProperty<>("djfs-client" +
            "-default_folders-supported-params",
            Cf.list(
                    "uid", "check_exist"
            ));

    private final MapF<String, DynamicProperty<ListF<String>>> supportedParametersByEndpoint = Cf.map(
            BULK_INFO_BY_RESOURCE_IDS_ENDPOINT, bulkInfoByResorceIdsSupportedParams,
            BULK_INFO_ENDPOINT, bulkInfoSupportedParams,
            INFO_ENDPOINT, infoSupportedParams,
            DEFAULT_FOLDERS_ENDPOINT, defaultFoldersSupportedParams
    );

    private final MapF<String, DynamicProperty<ListF<String>>> supportedMetasByEndpoint = Cf.map(
            BULK_INFO_BY_RESOURCE_IDS_ENDPOINT, bulkInfoSupportedMetaFields,
            BULK_INFO_ENDPOINT, bulkInfoSupportedMetaFields,
            INFO_ENDPOINT, infoSupportedMetaFields
    );

    public MpfsClientImpl(String host) {
        this(
                ApacheHttpClientUtils.multiThreadedClient(DEFAULT_TIMEOUT, DEFAULT_MAX_CONNECTIONS),
                ApacheHttpClientUtils.multiThreadedClient(DEFAULT_TIMEOUT, DEFAULT_MAX_CONNECTIONS),
                host, MpfsUserInitParamsExtractor.empty(), false, Option.empty(), Option.empty(), false);
    }

    public MpfsClientImpl(HttpClient httpClient, HttpClient longRequestsHttpClient,
                          String host, MpfsUserInitParamsExtractor userInitParams, boolean initOnDemand,
                          Option<String> djfsHost,
                          Option<HttpClient> djfsHttpClient, boolean ignoreUserNotInitializedException) {
        Option<Function1V<MpfsUser>> userInit = Option.of(this::userInit);
        if (!initOnDemand) {
            userInit = Option.empty();
        }

        this.mpfsHost = host;
        this.executor = new MpfsRequestExecutor(httpClient, longRequestsHttpClient, userInit);
        this.userInitParams = userInitParams;
        this.djfsHost = djfsHost;
        this.djfsHttpClient = djfsHttpClient;
        this.ignoreUserNotInitializedException = ignoreUserNotInitializedException;
    }

    private void userInit(MpfsUser uid) {
        executeAndGetJsonNode(uid, jsonUrl("user_init", uid,
                userInitParams.extract(uid).flatMap(t -> Cf.list(t.get1(), t.get2())).toArray()));
    }

    @Override
    public void userInit(MpfsUser uid, Language locale, String source, Option<String> b2b) {
        Tuple2List<String, String> params =
                Tuple2List.fromPairs("locale", locale.value(), "source", source)
                        .plus(b2b.map(organization -> Tuple2.tuple("b2b", organization)));
        executeAndGetJsonNode(uid, jsonUrl("user_init",
                uid, params.flatMap(t -> Cf.list(t.get1(), t.get2())).toArray()));
    }

    @Override
    public boolean needInit(MpfsUser uid) {
        return !executeAndGetJsonNode(uid, jsonUrl("user_check", uid)).get("need_init").textValue().equals("0");
    }

    @Override
    public MpfsCallbackResponse markMulcaIdToRemove(MulcaId mulcaId) {
        return executeAndGetResponse(Option.empty(), getMarkMulcaIdToRemoveUri(mulcaId));
    }

    @Override
    public MpfsCallbackResponse getFullRelativeTree(MpfsUser uid, String path) {
        return executeLongRequestAndGetResponse(Option.of(uid), getFullRelativeTreeUri(uid.getUidStr(), path));
    }

    @Override
    public MpfsCallbackResponse getFullRelativeTreePublic(String hash, boolean showNda) {
        return executeLongRequestAndGetResponse(Option.empty(), getFullRelativeTreePublicUri(hash, showNda));
    }

    @Override
    public MpfsCallbackResponse getAlbumResources(String publicKey, Option<MpfsUser> uid) {
        return executeLongRequestAndGetResponse(uid, getAlbumResourcesListUri(publicKey, uid.map(MpfsUser::getUidStr)));
    }

    @Override
    public MpfsCallbackResponse getFilesResources(String mpfsOid, MpfsUser uid) {
        return executeLongRequestAndGetResponse(uid, getFilesListUri(mpfsOid, uid.getUidStr()));
    }

    @Override
    public MpfsFileInfo getFileInfoByHid(MpfsHid mpfsHid) {
        MpfsCallbackResponse response = executeAndGetResponse(Option.empty(), getFileInfoUriByHid(mpfsHid));
        return MpfsFileInfo.parse(response.getResponse());
    }

    @Override
    public ListF<MpfsFileInfo> bulkInfoByResourceIds(MpfsUser uid, SetF<String> meta, ListF<String> resourceIds,
                                                     ListF<String> enableServiceIds) {
        UriData urlData = url(BULK_INFO_BY_RESOURCE_IDS_ENDPOINT, uid, Option.of(Cf.toList(meta)), Option.empty(),
                "enable_service_ids", enableServiceIds.mkString(","));

        String responseBody = urlData.isDjfs() ?
                postAndGetDjfsResponse(Option.of(uid), urlData.getPath(), jsonArrayBytest(resourceIds)).getResponse() :
                postAndGetResponse(Option.of(uid), urlData.getPath(), jsonArrayBytest(resourceIds)).getResponse();
        return MpfsFileInfo.parseList(responseBody);
    }

    @Override
    public ListF<MpfsFileInfo> bulkInfoByPaths(MpfsUser uid, SetF<String> meta, ListF<String> paths) {
        UriData uriData = url(BULK_INFO_ENDPOINT, uid, Option.of(Cf.toList(meta)), Option.of(paths));

        String responseBody = uriData.isDjfs() ?
                postAndGetDjfsResponse(Option.of(uid), uriData.getPath(), jsonArrayBytest(paths)).getResponse() :
                postAndGetResponse(Option.of(uid), uriData.getPath(), jsonArrayBytest(paths)).getResponse();
        return MpfsFileInfo.parseList(responseBody, meta);
    }

    private byte[] jsonArrayBytest(ListF<String> resourceIds) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        JsonWriter writer = new JacksonJsonWriterFactory().createJsonWriter(baos);

        writer.writeArrayStart();
        resourceIds.forEach(writer::writeString);
        writer.writeArrayEnd();
        writer.flush();

        return baos.toByteArray();
    }

    @Override
    public void streamingListByUidAndPath(MpfsUser uid, String path, Option<Integer> offset, Option<Integer> amount,
                                          Option<String> sort, boolean orderDesc,
                                          Function1V<Iterator<MpfsFileInfo>> callback) {
        String uri = listByUidAndPathUrl(uid, path, offset, amount, sort, orderDesc);

        executor.executeGetWithReinit(Option.of(uid), uri, new Abstract200ExtendedResponseHandler<Void>() {
            @Override
            protected Void handle200Response(HttpResponse response) throws IOException {
                JsonParser jsonParser = (new JsonFactory()).createParser(response.getEntity().getContent());
                ObjectMapper objectMapper = new ObjectMapper();

                JsonToken nextToken = jsonParser.nextToken();
                if (nextToken == JsonToken.START_OBJECT) {
                    JsonNode node = objectMapper.readTree(jsonParser);
                    callback.accept(Cf.list(MpfsFileInfo.parse(new JacksonJsonNodeWrapper(node))).iterator());
                    return null;
                }

                Check.equals(JsonToken.START_ARRAY, nextToken);
                jsonParser.nextToken();

                callback.accept(new Iterator<MpfsFileInfo>() {
                    @Override
                    public boolean hasNext() {
                        try {
                            if (jsonParser.currentToken() == null) {
                                jsonParser.nextToken();
                            }
                            return JsonToken.START_OBJECT == jsonParser.currentToken();
                        } catch (IOException e) {
                            throw ExceptionUtils.translate(e);
                        }
                    }

                    @Override
                    public MpfsFileInfo next() {
                        try {
                            JsonNode node = objectMapper.readTree(jsonParser);
                            return MpfsFileInfo.parse(new JacksonJsonNodeWrapper(node));
                        } catch (IOException e) {
                            throw ExceptionUtils.translate(e);
                        }
                    }
                });
                return null;
            }
        });
    }

    @Override
    public MpfsListResponse listByUidAndPath(MpfsUser uid, String path, Option<Integer> offset, Option<Integer> amount,
                                             Option<String> sort, boolean orderDesc) {
        String uri = listByUidAndPathUrl(uid, path, offset, amount, sort, orderDesc);
        String responseBody = executeAndGetResponse(uid, uri).getResponse();
        if (!responseBody.startsWith("[")) {
            responseBody = "[" + responseBody + "]";
        }
        ListF<MpfsFileInfo> elems = MpfsFileInfo.parseList(responseBody);
        return new MpfsListResponse(elems.first(), elems.drop(1));
    }

    private String listByUidAndPathUrl(MpfsUser uid, String path, Option<Integer> offset, Option<Integer> amount,
                                       Option<String> sort, boolean orderDesc) {
        MapF<String, String> params = Cf.hashMap();

        params.put("uid", uid.getUidStr());
        params.put("path", path);
        params.put("meta", "");

        offset.forEach(v -> params.put("offset", v.toString()));
        amount.forEach(v -> params.put("amount", v.toString()));
        sort.forEach(v -> params.put("sort", v));

        if (orderDesc) {
            params.put("order", "0");
        }

        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/list", params);
    }

    @Override
    public MpfsFileInfo getFileInfoByUidAndPath(MpfsUser uid, String path, MapF<String, Object> params,
                                                ListF<String> metaFields) {
        UriData uriData = getFileInfoUriByUidAndPath(uid.getUidStr(), path, params, metaFields);
        MpfsCallbackResponse response = uriData.isDjfs() ?
                executeAndGetDjfsResponse(uid, uriData.getPath()) :
                executeAndGetResponse(uid, uriData.getPath());
        return MpfsFileInfo.parse(response.getResponse());
    }

    @Override
    public MpfsFileInfo getFileInfoByUidAndPath(MpfsUser uid, String path, ListF<String> metaFields) {
        return getFileInfoByUidAndPath(uid, path, Cf.map(), metaFields);
    }

    @Override
    public void mkdir(MpfsUser uid, String path) {
        executeAndGetResponse(uid, mkdirUri(uid.getUidStr(), path));
    }

    /**
     * return path
     */
    @Override
    public String mksysdir(MpfsUser uid, String type) {
        return JsonNodeUtils
                .getNode(executeAndGetResponse(uid, mksysdirUri(uid.getUidStr(), type)).getResponse())
                .get("id").textValue();
    }

    @Override
    public Option<MpfsShareFolderInvite> getNotApprovedInvite(MpfsUser uid, String hash) {
        return getNotApprovedInvites(uid).find(i -> i.hash.equals(hash));
    }

    @Override
    public ListF<MpfsShareFolderInvite> getNotApprovedInvites(MpfsUser uid) {
        MpfsCallbackResponse resp = executeAndGetResponse(uid, jsonUrl("share_list_not_approved_folders", uid));

        return MpfsShareFolderInvite.P.parseListJson(resp.getResponse());
    }

    @Override
    public Option<MpfsOperation> getOperation(MpfsUser uid, String oid) {
        String url = url("desktop/status", uid, Option.empty(), Option.empty(), "oid", oid).getPath();
        return suppress404(() -> MpfsOperation.P.parseJson(executeAndGetResponse(uid, url).getResponse()));
    }

    @Override
    public ListF<MpfsOperation> activeOperations(MpfsUser uid) {
        String uri = UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/active_operations",
                Cf.map("uid", uid.getUidStr()));
        return MpfsOperation.P.parseListJson(executeAndGetResponse(uid, uri).getResponse());
    }

    @Override
    public MpfsStoreOperation astore(MpfsUser uid, String path, String md5, Tuple2List<String, String> headers) {
        String uri = UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/astore", Cf.<String, Object>map(
                "uid", uid.getUidStr(),
                "path", path,
                "md5", md5
        ));
        return MpfsStoreOperation.P.parseJson(executeAndGetResponse(uid, uri, headers).getResponse());
    }

    @Override
    public MpfsStoreOperation dstore(MpfsUser uid, String path, String md5, boolean iSwearIWillNeverPublishThisUrl) {
        int https = iSwearIWillNeverPublishThisUrl ? 0 : 1;
        String uri = jsonUrl("dstore", uid, "path", path, "md5", md5, "use_https", https);
        MpfsPatchOperation operation = MpfsPatchOperation.P.parseJson(executeAndGetResponse(uid, uri).getResponse());

        operation
                .getUploadUrl()
                .filterNot(url -> url.startsWith(iSwearIWillNeverPublishThisUrl ? "http" : "https"))
                .ifPresent(url -> logger.error(new IllegalStateException(
                        "url does not match requested type (use_https=" + https + "): " + url)));

        return operation;
    }

    @Override
    public void setprop(MpfsUser uid, String path, MapF<String, String> properties) {
        String uri = jsonUrl("setprop", uid, "path", path);
        uri = UrlUtils.addParameters(uri, properties);

        executeAndGetResponse(uid, uri);
    }

    @Override
    public void uploadEmptyFile(MpfsUser uid, String path, MapF<String, Object> additionalParams) {
        String uri = jsonUrl("store", uid,
                "path", path,
                "force", 0,
                "size", 0,
                "sha256", "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855",
                "md5", "d41d8cd98f00b204e9800998ecf8427e"
        );

        uri = UrlUtils.addParameters(uri, additionalParams);
        MpfsStoreOperation storeOp =
                MpfsStoreOperation.P.parseJson(executeAndGetResponse(uid, uri).getResponse());

        if (!storeOp.getStatus().isSome("hardlinked")) {
            //mustn"t be called ever
            HttpPut httpPut = new HttpPut(storeOp.getUploadUrl().get());
            ByteArrayEntity entity = new ByteArrayEntity(new byte[]{});
            httpPut.setEntity(entity);
            ApacheHttpClientUtils.execute(httpPut, new ReadStringResponseHandler());
        }
    }

    @Override
    public MpfsStoreOperation store(MpfsStoreOperationContext c, Tuple2List<String, String> headers) {
        MapF<String, Object> parameters = Cf.hashMap();

        if (c.additionalParams != null) {
            parameters.putAll(c.additionalParams);
        }
        parameters.put("uid", c.uid.getUidStr());
        parameters.put("path", c.path);
        // TODO: use 0 only if we are sure it will be local upload
        parameters.put("use_https", c.iSwearIWillNeverPublishThisUrl ? 0 : 1);

        if (c.force) {
            parameters.put("force", 1);
        }

        c.callback.forEach(v -> parameters.put("callback", v));
        c.md5.forEach(v -> parameters.put("md5", v));
        c.sha256.forEach(v -> parameters.put("sha256", v));
        c.size.forEach(v -> parameters.put("size", v));
        c.replaceMd5.forEach(v -> parameters.put("replace_md5", v));

        // iOS live-photo: CHEMODAN-40827
        c.livePhotoMd5.forEach(v -> parameters.put("live_photo_md5", v));
        c.livePhotoSha256.forEach(v -> parameters.put("live_photo_sha256", v));
        c.livePhotoType.forEach(v -> parameters.put("live_photo_type", v));
        c.livePhotoSize.forEach(v -> parameters.put("live_photo_size", v));
        c.livePhotoOperation.forEach(v -> parameters.put("live_photo_operation_id", v));

        c.photostreamDestination.forEach(v -> parameters.put("photostream_destination", v));

        c.deviceResourceSubtype.forEach(v -> parameters.put("device_resource_subtype", v));
        c.deviceCollections.forEach(v -> parameters.put("device_collections", v));
        c.deviceOriginalPath.forEach(v -> parameters.put("device_original_path", v));

        c.sourceId.forEach(v -> parameters.put("source_id", v));
        c.forceDeletionLogDeduplication.forEach(v -> parameters.put("force_deletion_log_deduplication", v));

        c.ctime.forEach(v -> parameters.put("ctime", v.getMillis() / 1000));
        c.mtime.forEach(v -> parameters.put("mtime", v.getMillis() / 1000));
        c.etime.forEach(v -> parameters.put("etime", v.getMillis() / 1000));

        if (c.isHidden) {
            parameters.put("visible", 0);
        }
        if (c.isScreenshot) {
            parameters.put("screenshot", 1);
        }
        if (c.isPublic) {
            parameters.put("public", 1);
        }

        String uri = UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/store", parameters);

        MpfsCallbackResponse callbackResponse = executeAndGetResponse(c.uid, uri, headers);

        MpfsStoreOperation operation =
                MpfsStoreOperation.P.parseJson(callbackResponse.getResponse());

        operation
                .getUploadUrl()
                .filterNot(url -> url.startsWith(c.iSwearIWillNeverPublishThisUrl ? "http" : "https"))
                .ifPresent(url -> logger.error(new IllegalStateException(
                        "url does not match requested type (use_https=" + parameters.getO("use_https") + "): " + url)));

        operation.setDiskPath(Option.ofNullable(callbackResponse.getHeaders().get(DISK_PATH_HEADER)));

        return operation;
    }

    @Override
    public MpfsCallbackResponse setPrivate(MpfsUser uid, String path) {
        return executeAndGetResponse(uid, jsonUrl("set_private", uid, "path", path));
    }

    @Override
    public MpfsCallbackResponse setPublic(MpfsUser uid, String path) {
        return setPublic(uid, path, Tuple2List.tuple2List());
    }

    @Override
    public MpfsCallbackResponse setPublic(MpfsUser uid, String path, Tuple2List<String, String> headers) {
        return executeAndGetResponse(uid, jsonUrl("set_public", uid, "path", path), headers);
    }

    @Override
    public String getPublicUrl(MpfsCallbackResponse response) {
        return JsonNodeUtils.getNode(response.getResponse()).get("short_url").asText();
    }

    @Override
    public MpfsPublicSettings getPublicSettings(MpfsUser uid, String path) {
        MpfsCallbackResponse response = executeAndGetResponse(uid,
                jsonUrl("get_public_settings", uid, "path", path));
        return MpfsPublicSettings.parse(response.getResponse());
    }

    @Override
    public MpfsPublicSettings getPublicSettingsByHash(MpfsUser uid, String hash) {
        MpfsCallbackResponse response = executeAndGetResponse(uid,
                jsonUrl("get_public_settings_by_hash", uid, "private_hash", hash));
        return MpfsPublicSettings.parse(response.getResponse());
    }

    @Override
    public MpfsCallbackResponse stateRemove(MpfsUser uid, String key) {
        return executeAndGetResponse(uid, jsonUrl("state_remove", uid, "key", key));
    }

    @Override
    public MpfsCallbackResponse stateSet(MpfsUser uid, String key, String value) {
        return executeAndGetResponse(uid, jsonUrl("state_set", uid, "key", key, "value", value));
    }

    @Override
    public MpfsCallbackResponse settingRemove(MpfsUser uid, String namespace, String key) {
        return executeAndGetResponse(uid, jsonUrl("setting_remove", uid, "namespace", namespace, "key", key));
    }

    @Override
    public MpfsCallbackResponse settingSet(MpfsUser uid, String namespace, String key, String value) {
        return executeAndGetResponse(uid,
                jsonUrl("setting_set", uid, "namespace", namespace, "key", key, "value", value));
    }

    @Override
    public MpfsCallbackResponse setProp(MpfsUser uid, String path, Tuple2List<String, String> set,
                                        ListF<String> remove) {
        ListF<String> params =
                Cf.list("path", path).plus(set.map(t -> Cf.list(t._1, t._2)).reduceRight(ListF::plus));

        if (remove.isNotEmpty()) {
            params = params.plus("setprop_delete", String.join(",", remove));
        }
        return executeAndGetResponse(uid, jsonUrl("setprop", uid, params.toArray()));
    }

    @Override
    public void userInstallDevice(MpfsUser uid, String type, String deviceId, MapF<String, Object> params) {
        String uri = jsonUrl("user_install_device", uid,
                "type", type,
                "id", deviceId);

        uri = UrlUtils.addParameters(uri, params);

        executeAndGetResponse(uid, uri);
    }

    @Override
    public <T> T diff(MpfsUser uid, String path, Option<Long> version,
                      Function<HttpResponse, T> indexResponseProcessor) {
        MapF<String, Object> parameters = Cf.hashMap();

        parameters.put("uid", uid.getUidStr());
        parameters.put("path", path);
        version.forEach(v -> parameters.put("version", v));

        String uri = UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/diff", parameters);

        return executor.executeGet(uri, indexResponseProcessor::apply);
    }

    @Override
    public MpfsCallbackResponse rm(MpfsUser uid, String path) {
        return executeAndGetResponse(uid, rmUri(uid, path));
    }

    @Override
    public MpfsOperation asyncRm(MpfsUser uid, String path, Option<String> ifMatch, String callbackUrl) {
        MpfsCallbackResponse response = executeAndGetResponse(
                uid, asyncRmUri("rm", uid.getUidStr(), path, ifMatch, callbackUrl));
        return MpfsOperation.P.parseJson(response.getResponse());
    }

    @Override
    public MpfsOperation asyncTrashAppend(MpfsUser uid, String path, Option<String> ifMatch, String callbackUrl) {
        MpfsCallbackResponse response = executeAndGetResponse(
                uid, asyncRmUri("trash_append", uid.getUidStr(), path, ifMatch, callbackUrl));
        return MpfsOperation.P.parseJson(response.getResponse());
    }

    @Override
    public MpfsOperation asyncTrashDropAll(MpfsUser uid, String callbackUrl) {
        String uri = jsonUrl("async_trash_drop_all", uid, "callback", callbackUrl);
        return MpfsOperation.P.parseJson(executeAndGetResponse(uid, uri).getResponse());
    }

    @Override
    public MpfsCallbackResponse copy(MpfsUser uid, String srcPath, String dstPath, boolean checkHidsBlockings) {
        if (srcPath.equals(dstPath)) {
            //XXX: very very bad
            throw new PermanentHttpFailureException("CopySame", HttpStatus.SC_409_CONFLICT);
        }
        return executeAndGetResponse(uid, copyUri(uid, srcPath, dstPath, checkHidsBlockings));
    }

    @Override
    public MpfsOperation asyncCopy(MpfsUser uid, String srcPath, String dstPath, boolean overwrite,
                                   String callbackUrl) {
        if (srcPath.equals(dstPath)) {
            //XXX: very very bad
            throw new PermanentHttpFailureException("CopySame", HttpStatus.SC_409_CONFLICT);
        }

        MpfsCallbackResponse response = executeAndGetResponse(
                uid, asyncCopyUri(uid, srcPath, dstPath, overwrite, callbackUrl));
        return MpfsOperation.P.parseJson(response.getResponse());
    }

    @Override
    public MpfsCallbackResponse move(MpfsUser uid, String srcPath, String dstPath, boolean checkHidsBlockings) {
        if (srcPath.equals(dstPath)) {
            //XXX: very very bad
            throw new PermanentHttpFailureException("MoveSame", HttpStatus.SC_409_CONFLICT);
        }
        return executeAndGetResponse(uid, moveUri(uid, srcPath, dstPath, checkHidsBlockings));
    }

    @Override
    public MpfsOperation asyncMove(MpfsUser uid, String srcPath, String dstPath, boolean overwrite,
                                   String callbackUrl) {
        if (srcPath.equals(dstPath)) {
            //XXX: very very bad
            throw new PermanentHttpFailureException("MoveSame", HttpStatus.SC_409_CONFLICT);
        }
        MpfsCallbackResponse response = executeAndGetResponse(
                uid, asyncMoveUri(uid, srcPath, dstPath, overwrite, callbackUrl));
        return MpfsOperation.P.parseJson(response.getResponse());
    }

    @Override
    public Option<MpfsFileInfo> getFileInfoOByUidAndPath(MpfsUser uid, String path, ListF<String> metaFields) {
        return suppress404(() -> getFileInfoByUidAndPath(uid, path, metaFields));
    }

    @Override
    public Option<MpfsFileInfo> getFileInfoOByFileId(MpfsUser owner, String fileId) {
        return getFileInfoOByFileId(owner, owner.getUidStr(), fileId);
    }

    @Override
    public Option<MpfsFileInfo> getFileInfoOByFileId(MpfsUser uid, String owner, String fileId) {
        return suppress404(() -> getFileInfoByFileId(uid, owner, fileId));
    }

    @Override
    public MpfsFileInfo getFileInfoByFileId(MpfsUser owner, String fileId) {
        return getFileInfoByFileId(owner, owner.getUidStr(), fileId);
    }

    @Override
    public MpfsFileInfo getFileInfoByFileId(MpfsUser uid, String owner, String fileId) {
        MpfsCallbackResponse response =
                executeAndGetResponse(uid, getFileInfoUriByFileId(uid.getUidStr(), owner, fileId));
        return MpfsFileInfo.parse(response.getResponse());
    }

    @Override
    public MpfsFileInfo getFileInfoByCommentId(MpfsUser uid, String entityType, String entityId) {
        MpfsCallbackResponse response =
                executeAndGetResponse(uid, getFileInfoUriByCommentId(uid, entityType, entityId));
        return MpfsFileInfo.parse(response.getResponse());
    }

    @Override
    public MpfsFileInfo getPublicInfo(String hash, Option<MpfsUser> uid) {
        String uri = jsonUrl("public_info", uid.getOrElse(MpfsUser.of(0)), "private_hash", hash, "meta", "");
        JsonNode jsonNode = executeAndGetJsonNode(uid, uri);
        return MpfsFileInfo.parse(new JacksonJsonNodeWrapper(jsonNode.get("resource")));
    }

    @Override
    public Option<MpfsFileInfo> getFileInfoOByPrivateHash(String privateHash) {
        return suppress404(() -> MpfsFileInfo.parse(new JacksonJsonNodeWrapper(
                executeAndGetJsonNode(Option.empty(), getResourceInfoUri(privateHash)).get("resource"))));
    }

    @Override
    public Option<MpfsCommentsPermissions> getCommentsPermissions(MpfsUser uid, String entityType, String entityId) {
        return suppress404(() -> MpfsCommentsPermissions.ps.getParser().parseJson(
                executeAndGetResponse(uid, getCommentsPermissionsUri(uid, entityType, entityId)).getResponse()));
    }

    @Override
    public Option<String> getSharedFolderPathByUidAndGroupId(MpfsUser uid, String groupId) {
        return suppress404(() -> executeAndGetJsonNode(uid, getShareFolderInfoUri(uid, groupId)).get("id").textValue());
    }

    @Override
    public String getPublicFileAddress(String privateHash) {
        return executeAndGetResponse(Option.empty(), getPublicFileAddressUri(privateHash)).getResponse();
    }

    @Override
    public MpfsUserInfo getUserInfoObj(MpfsUser uid) {
        return MpfsUserInfo.P.parseJson(executeAndGetResponse(uid, getUserInfoUri(uid.getUidStr())).getResponse());
    }

    @Override
    public ListF<MpfsUserServiceInfo> getUserServiceInfoList(MpfsUser uid) {
        return MpfsUserServiceInfo.P.parseListJson(
                executeAndGetResponse(uid, getServiceListUri(uid.getUidStr())).getResponse());
    }

    @Override
    public MapF<String, String> getDefaultFolders(MpfsUser uid) {
        MapF<String, Object> parameters = Cf.map("uid", uid.getUidStr());
        UriData uriData = getEndpointPathData(Option.empty(), parameters, DEFAULT_FOLDERS_ENDPOINT, Option.empty());
        String uri = UrlUtils.addParameters(uriData.getPath(), parameters);
        MpfsCallbackResponse response = uriData.isDjfs() ?
                executeAndGetDjfsResponse(uid, uri) :
                executeAndGetResponse(uid, uri);
        ObjectNode node = (ObjectNode) JsonNodeUtils.getNode(response.getResponse());
        return Cf.x(node.fieldNames()).toList().toMapMappingToValue(field -> node.get(field).asText());
    }

    @Override
    public String getUserInfo(MpfsUser uid) {
        return executeAndGetResponse(uid, getUserInfoUri(uid.getUidStr())).getResponse();
    }

    @Override
    public Option<MapF<String, MapF<String, String>>> getUserSettingsO(MpfsUser uid) {
        return suppress404(() ->
                Cf.x(executeAndGetJsonNode(uid, getUserInfoUri(uid.getUidStr())).get("settings").fields())
                        .map(Tuple2.join(Map.Entry::getKey, Map.Entry::getValue))
                        .map(Tuple2.map2F(ns -> Cf.x(ns.fields())
                                .toList().toMap(Map.Entry::getKey, s -> s.getValue().asText())))
                        .toList().toMap(Tuple2::get1, Tuple2::get2));
    }

    @Override
    public MpfsCallbackResponse getOfficeStoreInfo(String resourceId, String accessToken, String accessTokenTtl,
                                                   MapF<String, String> headers) {
        String url = getOfficeStoreUri(resourceId, accessToken, accessTokenTtl);
        return postAndGetResponse(url, serializeHeadersToJson(headers));
    }

    @Override
    public Option<MpfsGroupUids> getShareUidsInGroupO(MpfsUser owner, String fileId) {
        return suppress404(() -> getShareUidsInGroup(owner, owner.getUidStr(), fileId));
    }

    @Override
    public MpfsGroupUids getShareUidsInGroup(MpfsUser uid, String owner, String fileId) {
        MpfsCallbackResponse response = executeAndGetResponse(uid, getShareUidsInGroupUri(uid, owner, fileId));
        return MpfsGroupUids.PS.getParser().parseJson(response.getResponse());
    }

    @Override
    public Option<MpfsLentaBlockFileIds> getLentaBlocksFileIds(
            MpfsUser uid, String resourceId, String mediaType,
            MpfsUid modifier, int mtimeGte, int mtimeLte, int amount) {
        return suppress404(() -> {
            JsonNode node = executeAndGetJsonNode(uid, getLentaBlocksFileIdsUri(
                    uid, resourceId, mediaType, modifier, mtimeGte, mtimeLte, amount));

            ListF<JsonNode> elements = Cf.x(node.elements()).toList();

            return new MpfsLentaBlockFileIds(
                    elements.get(0).get("meta").get("total_results_count").intValue(),
                    elements.drop(1).map(e -> e.get("meta").get("file_id").textValue()));

        });
    }

    @Override
    public Option<MpfsLentaBlockFullDescription> getLentaBlockFilesData(
            MpfsUser uid, String resourceId, String mediaType,
            MpfsUid modifier, int mtimeGte, int mtimeLte, int amount) {
        return getLentaBlockFilesData(uid, resourceId, mediaType, modifier, mtimeGte, mtimeLte, amount,
                "total_results_count,file_id,resource_id,preview,file_url,folder_url");
    }

    @Override
    public Option<MpfsLentaBlockFullDescription> getLentaBlockFilesData(
            MpfsUser uid, String resourceId, String mediaType,
            MpfsUid modifier, int mtimeGte, int mtimeLte, int amount, String meta) {
        return suppress404(() -> {
            JsonNode node = executeAndGetJsonNode(uid, getLentaBlockDataUri(
                    uid, resourceId, mediaType, modifier, mtimeGte, mtimeLte, amount, meta));

            return buildresultMpfsLenta(node, resourceId);
        });
    }

    MpfsLentaBlockFullDescription buildresultMpfsLenta(JsonNode node, String resourceId) {
        ListF<JsonNode> elements = Cf.x(node.elements()).toList();

        JsonNode lentaBlock = elements.get(0);
        Validate.notNull(lentaBlock, "Can't fetch lentaBlock from " + node);
        return new MpfsLentaBlockFullDescription(
                MpfsLentaBlockItemDescription.
                        getSubNodeO(lentaBlock, "meta", "total_results_count").get().asInt(),
                MpfsLentaBlockItemDescription.cons(lentaBlock, resourceId),
                elements.drop(1).map(MpfsLentaBlockItemDescription::cons));
    }

    private UriData url(String operation, MpfsUser uid, Option<ListF<String>> metaFields,
                        Option<ListF<String>> resourcePaths,
                        Object... params) {
        Validate.isTrue(params.length % 2 == 0);

        MapF<String, Object> parameters = Cf.hashMap();

        parameters.put("uid", uid.getUidStr());

        for (int i = 0; i < params.length; i += 2) {
            parameters.put(params[i].toString(), params[i + 1].toString());
        }

        UriData uriData = getEndpointPathData(metaFields, parameters, operation, resourcePaths);
        return new UriData(
                uriData.isDjfs(),
                UrlUtils.addParameters(
                        uriData.getPath(),
                        metaFields.map(fields -> parameters.plus1("meta", fields.mkString(","))).getOrElse(parameters)
                )
        );
    }

    private String getHostWithProtocol(String host) {
        if (!host.startsWith("http://") && !host.startsWith("https://")) {
            return "https://" + host;
        }
        return host;
    }

    private String jsonUrl(String operation, MpfsUser uid, Object... params) {
        return url("json/" + operation, uid, Option.empty(), Option.empty(), params).getPath();
    }

    @Override
    public String getMarkMulcaIdToRemoveUri(MulcaId mulcaId) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/service/mark_stid_deleted",
                "stid", mulcaId.toSerializedString());
    }

    @Override
    public String getFullRelativeTreeUri(String uid, String path) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/json/fulltree",
                "uid", uid, "path", path, "meta", META_TAGS);
    }

    @Override
    public String getFullRelativeTreePublicUri(String hash, boolean showNda) {
        String url = UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/json/public_fulltree",
                "private_hash", hash, "meta", META_TAGS);
        if (showNda) {
            url = UrlUtils.addParameter(url, "show_nda", "kladun");
        }
        return url;
    }

    @Override
    public String getAlbumResourcesListUri(String publicKey, Option<String> uid) {
        String url = UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/json/public_album_resources_list",
                "public_key", publicKey, "meta", META_TAGS);
        if (uid.isPresent()) {
            url = UrlUtils.addParameter(url, "uid", uid.get());
        }
        return url;
    }

    @Override
    public String getFilesListUri(String mpfsOid, String uid) {
        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/bulk_download_list",
                Cf.<String, String>map()
                        .plus1("oid", mpfsOid)
                        .plus1("uid", uid)
                        .plus1("meta", META_TAGS));
    }

    @Override
    public String getOfficeStoreUri(String resourceId, String accessToken, String accessTokenTtl) {
        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/office_store",
                Cf.<String, Object>map("resource_id", resourceId)
                        .plus1("access_token", accessToken)
                        .plus1("access_token_ttl", accessTokenTtl));
    }

    @Override
    public String getKladunDownloadCounterIncUri(String hash, long bytes) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/service/kladun_download_counter_inc",
                "hash", hash, "bytes", bytes);
    }

    @Override
    public String getCommentsPermissionsUri(MpfsUser uid, String entityType, String entityId) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/json/comments_permissions",
                "uid", uid.getUidStr(), "entity_type", entityType, "entity_id", entityId);
    }

    @Override
    public String getShareFolderInfoUri(MpfsUser uid, String groupId) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/json/share_folder_info",
                "uid", uid.getUidStr(), "gid", groupId);
    }

    @Override
    public String getFileInfoUriByHid(MpfsHid hid) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/service/hardlink", "hid", hid);
    }

    private String mkdirUri(String uid, String path) {
        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/mkdir",
                Cf.<String, Object>map("uid", uid)
                        .plus1("path", path));
    }

    private String mksysdirUri(String uid, String type) {
        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/mksysdir",
                Cf.<String, Object>map("uid", uid)
                        .plus1("type", type));
    }

    private String rmUri(MpfsUser uid, String path) {
        MapF<String, Object> parameters = Cf.hashMap();
        parameters.put("uid", uid.getUidStr());
        parameters.put("path", path);

        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/rm", parameters);
    }

    private String asyncRmUri(String op, String uid, String path, Option<String> ifMatch, String callbackUrl) {
        MapF<String, Object> parameters = Cf.hashMap();

        parameters.put("uid", uid);
        parameters.put("path", path);
        parameters.put("callback", callbackUrl);

        ifMatch.forEach(md5 -> parameters.put("md5", md5));

        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/async_" + op, parameters);
    }

    private String copyUri(MpfsUser uid, String srcPath, String dstPath, boolean checkHidsBlockings) {
        return copyOrMoveUri("copy", false, uid, srcPath, dstPath, false, Option.of(checkHidsBlockings), "");
    }

    private String moveUri(MpfsUser uid, String srcPath, String dstPath, boolean checkHidsBlockings) {
        return copyOrMoveUri("move", false, uid, srcPath, dstPath, false, Option.of(checkHidsBlockings), "");
    }

    private String asyncCopyUri(MpfsUser uid, String srcPath, String dstPath, boolean overwrite, String callbackUrl) {
        return copyOrMoveUri("copy", true, uid, srcPath, dstPath, overwrite, Option.empty(), callbackUrl);
    }

    private String asyncMoveUri(MpfsUser uid, String srcPath, String dstPath, boolean overwrite, String callbackUrl) {
        return copyOrMoveUri("move", true, uid, srcPath, dstPath, overwrite, Option.empty(), callbackUrl);
    }

    private String copyOrMoveUri(String op, boolean async, MpfsUser uid, String srcPath, String dstPath,
                                 boolean overwrite, Option<Boolean> checkHidsBlockings,
                                 String callbackUrl) {
        MapF<String, Object> parameters = Cf.hashMap();

        parameters.put("uid", uid.getUidStr());
        parameters.put("src", srcPath);
        parameters.put("dst", dstPath);
        if (async) {
            parameters.put("callback", callbackUrl);
        }
        if (overwrite) {
            parameters.put("force", 1);
        }
        if (checkHidsBlockings.isPresent()) {
            parameters.put("check_hids_blockings", checkHidsBlockings.get() ? "1" : "0");
        }

        String action = (async ? "async_" : "") + op;

        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/" + action, parameters);
    }

    @Override
    public UriData getFileInfoUriByUidAndPath(String uid, String path, MapF<String, Object> params,
                                              ListF<String> metaFields) {
        UriData uriData = getEndpointPathData(Option.of(metaFields), params, INFO_ENDPOINT, Option.of(Cf.list(path)));
        return new UriData(uriData.isDjfs(),
                UrlUtils.addParameters(uriData.getPath(),
                        params
                                .plus1("uid", uid)
                                .plus1("path", path)
                                .plus1("meta", metaFields.mkString(","))
                ));
    }

    private UriData getEndpointPathData(Option<ListF<String>> metaFields, MapF<String, Object> params, String path,
                                        Option<ListF<String>> resourcePaths) {
        if (!djfsHost.isPresent() || !djfsHttpClient.isPresent()) {
            return new UriData(false, addPath(getHostWithProtocol(mpfsHost), path));
        }
        if (resourcePaths
                .map(paths -> paths
                        .exists(resourcePath -> !djfsSupportedRootFolders.get().exists(rootFolder -> resourcePath.startsWith(rootFolder)))
                ).getOrElse(Boolean.FALSE)) {
            return new UriData(false, addPath(getHostWithProtocol(mpfsHost), path));
        }
        if (!supportedParametersByEndpoint.getO(path)
                .map(parameters -> parameters.get().containsAllTs(params.keys())).getOrElse(Boolean.FALSE)) {
            return new UriData(false, addPath(getHostWithProtocol(mpfsHost), path));
        }
        if (metaFields.isPresent() && (metaFields.get().isEmpty() ||
                !supportedMetasByEndpoint.getO(path)
                        .map(supportedMetaFields -> supportedMetaFields.get().containsAllTs(metaFields.get()))
                        .getOrElse(Boolean.FALSE))) {
            return new UriData(false, addPath(getHostWithProtocol(mpfsHost), path));
        }
        return new UriData(true, addPath(getDjfsEndpointSubpath(djfsHost.get()), path));
    }

    private String addPath(String subpath, String path) {
        return subpath + "/" + path;
    }

    @Override
    public String getFileInfoUriByFileId(String uid, String owner, String fileId) {
        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/info_by_file_id", Cf.<String, Object>map()
                .plus1("uid", uid)
                .plus1("owner_uid", owner)
                .plus1("file_id", fileId)
                .plus1("meta", ""));
    }

    @Override
    public String getFileInfoUriByCommentId(MpfsUser uid, String entityType, String entityId) {
        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/info_by_comment_id",
                Cf.<String, Object>map()
                .plus1("uid", uid.getUidStr())
                .plus1("entity_type", entityType)
                .plus1("entity_id", entityId)
                .plus1("meta", ""));
    }

    @Override
    public String getResourceInfoUri(String privateHash) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/support/public_info",
                "private_hash", privateHash, "meta", RESOURCE_INFO_META_TAGS);
    }

    @Override
    public String getPublicFileAddressUri(String privateHash) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/support/get_public_file_address",
                "private_hash", privateHash, "meta", "");
    }

    @Override
    public String getUserInfoUri(String uid) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/support/user_info", "uid", uid);
    }

    @Override
    public String getServiceListUri(String uid) {
        return UrlUtils.addParameter(getHostWithProtocol(mpfsHost) + "/billing/service_list", "uid", uid, "ip", "127.0.0.1");
    }

    @Override
    public String getShareUidsInGroupUri(MpfsUser uid, String owner, String fileId) {
        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/share_uids_in_group", Cf.<String,
                Object>map()
                .plus1("uid", uid.getUidStr())
                .plus1("owner_uid", owner)
                .plus1("file_id", fileId));
    }

    @Override
    public String getLentaBlocksFileIdsUri(
            MpfsUser uid, String resourceId, String mediaType,
            MpfsUid modifier, int mtimeGte, int mtimeLte, int amount) {
        return getLentaBlockDataUri(uid, resourceId, mediaType, modifier, mtimeGte, mtimeLte, amount,
                "total_results_count,file_id");
    }

    @Override
    public String getLentaBlockDataUri(
            MpfsUser uid, String resourceId, String mediaType,
            MpfsUid modifier, int mtimeGte, int mtimeLte, int amount, String metaFields) {
        return UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/lenta_block_list", Cf.<String, Object>map()
                .plus1("uid", uid.getUidStr())
                .plus1("resource_id", resourceId)
                .plus1("media_type", mediaType)
                .plus1("modify_uid", modifier.getRawValue())
                .plus1("mtime_gte", mtimeGte)
                .plus1("mtime_lte", mtimeLte)
                .plus1("amount", amount)
                .plus1("meta", metaFields));
    }

    static <T> Option<T> suppress404(Function0<T> requestF) {
        return suppressByCodes(MpfsClientImpl.suppressedCodes, requestF);
    }

    public static <T> Option<T> suppressByCodes(List<Integer> httpCodes, Function0<T> requestF) {
        try {
            return Option.of(requestF.apply());

        } catch (PermanentHttpFailureException e) {
            if (!e.getStatusCode().isMatch(httpCodes::contains)) {
                throw e;
            }
            return Option.empty();
        }
    }

    public static <T> Option<T> suppress4xx(Function0<T> requestF) {
        try {
            return Option.of(requestF.apply());

        } catch (PermanentHttpFailureException e) {
            if (e.getStatusCode().isPresent() && e.getStatusCode().get() >= 400 && e.getStatusCode().get() < 500) {
                return Option.empty();
            }
            throw e;
        }
    }

    public static void suppressByCodes(List<Integer> httpCodes, Function0V requestF) {
        try {
            requestF.apply();

        } catch (PermanentHttpFailureException e) {
            if (!e.getStatusCode().isMatch(httpCodes::contains)) {
                throw e;
            }
            logger.info("Error {} with body {} suppressed", e.getStatusCode(), e.responseBody);
        }
    }

    private JsonNode executeAndGetJsonNode(MpfsUser uid, String uriStr) {
        return executeAndGetJsonNode(Option.of(uid), uriStr);
    }

    private JsonNode executeAndGetJsonNode(Option<MpfsUser> uid, String uriStr) {
        try {
            return objectMapper.readTree(executeAndGetResponse(uid, uriStr).getResponse());
        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
    }

    private MpfsCallbackResponse executeAndGetResponse(MpfsUser uid, String uriStr,
                                                       Tuple2List<String, String> headers) {
        return executor.executeAndGetResponse(Option.of(uid), uriStr, headers);
    }

    private MpfsCallbackResponse executeAndGetResponse(MpfsUser uid, String uriStr) {
        return executeAndGetResponse(Option.of(uid), uriStr);
    }

    private MpfsCallbackResponse executeAndGetDjfsResponse(MpfsUser uid, String uriStr) {
        HttpClient client = djfsHttpClient.getOrThrow(IllegalStateException::new);
        URI uri = URI.create(uriStr);

        HttpGet get = MpfsRequestExecutor.withYcridHeaderSet(new HttpGet(uri));

        MpfsCallbackResponse response =
                ApacheHttpClientUtils.execute(get, client, new MpfsCallbackResponseHandler(uri));

        return processMpfsResponseStatusCode(Option.of(uid), response);
    }

    private MpfsCallbackResponse processMpfsResponseStatusCode(Option<MpfsUser> uid, MpfsCallbackResponse response) {
        int statusCode = response.getStatusCode();
        if (statusCode == 400 &&
                MpfsRequestExecutor.isErrorCodeEquals(response, MpfsRequestExecutor.USER_NOT_INITIALIZED_CODE) &&
                !ignoreUserNotInitializedException) {
            throw new UserNotInitializedException(uid.getOrElse(MpfsUser.of(0)));
        }
        if (statusCode == 403 && MpfsRequestExecutor.isErrorCodeEquals(response,
                MpfsRequestExecutor.USER_BLOCKED_CODE)) {
            throw new UserBlockedException(uid.getOrElse(MpfsUser.of(0)));
        }
        if (HttpStatus.is2xx(statusCode)) {
            return response;
        }
        if (HttpStatus.is4xx(statusCode)) {
            throw new PermanentHttpFailureException(response.getResponse(), response.toString(), statusCode,
                    response.getHeaders());
        }
        throw new HttpException(statusCode, response.getResponse(), response.toString());
    }

    MpfsCallbackResponse executeAndGetResponse(Option<MpfsUser> uid, String uriStr) {
        return executor.executeAndGetResponse(uid, uriStr);
    }

    private MpfsCallbackResponse executeLongRequestAndGetResponse(MpfsUser uid, String uriStr) {
        return executeLongRequestAndGetResponse(Option.of(uid), uriStr);
    }

    private MpfsCallbackResponse executeLongRequestAndGetResponse(Option<MpfsUser> uid, String uriStr) {
        return executor.executeLongRequestAndGetResponse(uid, uriStr);
    }

    private MpfsCallbackResponse postAndGetResponse(Option<MpfsUser> uid, String uriStr, byte[] content) {
        return executor.postAndGetResponse(uid, uriStr, content);
    }

    private MpfsCallbackResponse executeAndGetResponse(Option<MpfsUser> uid, String uriStr,
                                                       Tuple2List<String, String> headers, byte[] content) {
        return executor.executeAndGetResponse(uid, uriStr, headers, content);
    }

    public MpfsCallbackResponse postAndGetDjfsResponse(Option<MpfsUser> uid, String uriStr, byte[] content) {
        final MpfsCallbackResponse response = postAndGetDjfsResponse(uriStr, content);
        return processMpfsResponseStatusCode(uid, response);
    }

    private MpfsCallbackResponse postAndGetResponse(String uriStr, byte[] content) {
        return executor.postAndGetResponse(uriStr, content);
    }

    public MpfsCallbackResponse postAndGetDjfsResponse(String uriStr, byte[] content) {
        URI uri = URI.create(uriStr);

        HttpPost post = MpfsRequestExecutor.withYcridHeaderSet(new HttpPost(uri));
        post.setEntity(new ByteArrayEntity(content));
        post.setHeader(Header.ContentType.toString(), "application/json");
        return ApacheHttpClientUtils.execute(post, djfsHttpClient.getOrThrow(IllegalStateException::new),
                new MpfsCallbackResponseHandler(uri));
    }

    private MpfsCallbackResponse postAndGetResponse(
            String uriStr, byte[] content, Tuple2List<String, String> headers) {
        return executor.postAndGetResponse(uriStr, content, headers);
    }

    private byte[] serializeHeadersToJson(MapF<String, String> headers) {
        MapF<String, JsonValue> values = headers.mapValues(JsonString::valueOf);
        JsonValue root = new JsonObject(Cf.map("headers", new JsonObject(values)));

        return new JsonSerializer().serialize(root).getBytes(CharsetUtils.UTF8_CHARSET);
    }

    // For tests only
    @Override
    public void setMpfsHost(String mpfsHost) {
        EnvironmentType env = EnvironmentType.getActive();
        Validate.isTrue(env == EnvironmentType.DEVELOPMENT || env == EnvironmentType.TESTS);
        this.mpfsHost = mpfsHost;
    }

    @Override
    public boolean isMpfsHost(String host) {
        return mpfsHost.equals(host);
    }

    @Override
    public void close() throws IOException {
        executor.close();
    }

    @Override
    public MpfsCallbackResponse shareInviteUser(MpfsUser uid, String realPath, int rightsInt, String login) {
        String uri = jsonUrl("share_invite_user", uid,
                "path", realPath,
                "rights", rightsInt,
                "universe_service", "email",
                "universe_login", login + "@yandex.ru"
        );

        return executeAndGetResponse(uid, uri);
    }

    @Override
    public void rejectInvite(MpfsUser uid, String hash) {
        executeAndGetResponse(uid, jsonUrl("share_reject_invite", uid, "hash", hash));
    }

    @Override
    public String acceptInvite(MpfsUser uid, String hash) {
        JsonNode jsonNode = executeAndGetJsonNode(uid, jsonUrl("share_activate_invite", uid, "hash", hash));
        return jsonNode.get("id").asText();
    }

    @Override
    public int mobileSubscribe(MpfsUser uid, String token, Option<String> allow, String resources) {
        ListF<String> params = Cf.list("token", token).plus(allow.map(v -> Cf.list("allow", v)).getOrElse(Cf::list));
        String url = url("service/mobile_subscribe", uid, Option.empty(), Option.empty(), params.toArray()).getPath();
        String body = resources.isEmpty() ? "" : "resources=" + UrlUtils.urlEncode(resources);

        MpfsCallbackResponse response = postAndGetResponse(url, body.getBytes(),
                Tuple2List.fromPairs("Content-Type", "application/x-www-form-urlencoded"));
        int statusCode = response.getStatusCode();
        if (statusCode == HttpStatus.SC_400_BAD_REQUEST) {
            // Rewrite `Xiva filter size limit exceeded` to 503 code
            try {
                JsonNode code = JsonNodeUtils.getNode(response.getResponse()).path("code");
                if (!code.isMissingNode() && XIVA_LIMIT_EXCEEDED.equals(code.asText())) {
                    statusCode = HttpStatus.SC_503_SERVICE_UNAVAILABLE;
                }
            } catch (RuntimeException e) {
                logger.error("Failed to parse response body for 400: {}", e);
            }
        }
        return statusCode;
    }

    @Override
    public int mobileUnsubscribe(MpfsUser uid, String token) {
        return postAndGetResponse(url("service/mobile_unsubscribe", uid, Option.empty(), Option.empty(), "token",
                token).getPath(), "".getBytes())
                .getStatusCode();
    }

    @Override
    public String generateZaberunUrl(
            String stid,
            String fileName,
            String urlType,
            Option<String> uid,
            Option<String> md5,
            Option<String> contentType,
            Option<String> hash,
            Option<String> parser,
            Option<String> hid,
            Option<String> mediaType,
            Option<String> size,
            Option<Integer> limit,
            Option<Integer> fsize,
            Option<Integer> expireSeconds,
            Option<Integer> crop,
            Option<Boolean> inline,
            Option<Boolean> eternal) {
        MapF<String, String> params = Cf.<String, Option<?>>map()
                .plus1("uid", uid)
                .plus1("md5", md5)
                .plus1("content_type", contentType)
                .plus1("hash", hash)
                .plus1("parser", parser)
                .plus1("hid", hid)
                .plus1("media_type", mediaType)
                .plus1("size", size)
                .plus1("limit", limit)
                .plus1("fsize", fsize)
                .plus1("expire_seconds", expireSeconds)
                .plus1("crop", crop)
                .plus1("inline", inline)
                .plus1("eternal", eternal)
                .filterValues(Option::isPresent)
                .mapValues(Option::get)
                .mapValues(Object::toString)
                .plus1("stid", stid)
                .plus1("file_name", fileName)
                .plus1("url_type", urlType);
        String uri = UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/generate_zaberun_url", params);
        return executeAndGetJsonNode(Option.empty(), uri).get("zaberun_url").asText();
    }

    @Override
    public boolean isQuickMoveEnabled(String uid) {
        MpfsUser mpfsUser = MpfsUser.of(uid);
        String url = url("/service/check_reindexed_for_quick_move", mpfsUser, Option.empty(), Option.empty()).getPath();

        JsonNode jsonNode = executeAndGetJsonNode(mpfsUser, url);

        Check.notNull(jsonNode.get("result"), "result");
        Check.notNull(jsonNode.get("result").get("is_reindexed"), "result.is_reindexed");
        Check.isTrue(jsonNode.get("result").get("is_reindexed").isBoolean(), "not boolean result.is_reindexed");

        return jsonNode.get("result").get("is_reindexed").asBoolean();
    }

    @Override
    public MpfsCallbackResponse initNotes(MpfsUser uid, String src, String dst, MapF<String, Object> metaParams) {

        MapF<String, Object> parameters = Cf.hashMap();
        parameters.putAll(metaParams);
        parameters.put("uid", uid.getUidStr());
        parameters.put("src", src);
        parameters.put("dst", dst);

        String uri = UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/notes_init", parameters);

        return executeAndGetResponse(uid, uri);
    }

    @Override
    public Option<String> getFotkiAlbumItemUrl(PassportUid ownerUid, int albumId, String path) {
        MapF<String, Object> parameters = Cf.hashMap();
        parameters.put("owner_uid", ownerUid);
        parameters.put("fotki_album_id", albumId);
        parameters.put("path", path);

        String uri =
                UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/fotki_album_item_public_url", parameters);

        return suppress404(() -> executeAndGetJsonNode(MpfsUser.of(0), uri).get("url").asText())
                .filterNot(String::isEmpty);
    }

    @Override
    public Option<String> getFotkiAlbumUrl(PassportUid ownerUid, int albumId) {
        MapF<String, Object> parameters = Cf.hashMap();
        parameters.put("owner_uid", ownerUid);
        parameters.put("fotki_album_id", albumId);

        String uri = UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/json/fotki_album_public_url", parameters);

        return suppress404(() -> executeAndGetJsonNode(MpfsUser.of(0), uri).get("url").asText())
                .filterNot(String::isEmpty);
    }

    @Override
    public boolean arePhotosliceAlbumsEnabledSafe(MpfsUser uid) {
        try {
            return getUserInfoObj(uid).getPhotosliceAlbumsEnabled().isSome(1);
        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            logger.error("Couldn't get and albums enabled: {}", uid, e);
        }
        return false;
    }

    @Override
    public void processInappReceipt(PassportUid uid, String packageName, String storeType, String currency,
                                    String receipt) {
        MapF<String, Object> parameters = Cf.hashMap();
        parameters.put("uid", uid.toString());
        parameters.put("package_name", packageName);
        parameters.put("store_id", storeType);
        parameters.put("currency", currency);

        Tuple2List<String, String> headers = Tuple2List.fromPairs(Header.ContentType.toString(), "application/json");
        String uri =
                UrlUtils.addParameters(getHostWithProtocol(mpfsHost) + "/billing/process_receipt", parameters);
        byte[] body = JsonNodeUtils.getNode(Cf.map("receipt", receipt)).toString().getBytes(StandardCharsets.UTF_8);
        executeAndGetResponse(Option.of(MpfsUser.of(uid)), uri, headers, body);
    }

    private String getDjfsEndpointSubpath(String djfsHost) {
        return getHostWithProtocol(djfsHost) + "/api/legacy";
    }

    @Override
    public Map<String, Boolean> getFeatureToggles(MpfsUser uid) {
        JsonNode jsonNode = executeAndGetJsonNode(uid, jsonUrl("user_feature_toggles", uid));

        Map<String, Object> features = objectMapper.convertValue(jsonNode, new TypeReference<Map<String, Object>>() {});
        Map<String, Boolean> result = new HashMap<>();

        for (String featureName : features.keySet()) {
            boolean featureEnabled = features.get(featureName) instanceof Map &&
                    ((Map<?, ?>) features.get(featureName)).get("enabled") == Boolean.TRUE;
            result.put(featureName, featureEnabled);
        }

        return result;
    }
}
