package ru.yandex.chemodan.mpfs;

import java.io.IOException;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpUriRequest;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockFullDescription;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockItemDescription;
import ru.yandex.chemodan.mulca.MulcaClientContextConfiguration;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.chemodan.util.test.HttpRecorderRule;
import ru.yandex.chemodan.util.test.MpfsMockConfiguration;
import ru.yandex.chemodan.util.test.TestUser;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.parse.BenderParser;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.uri.Uri2;
import ru.yandex.misc.test.Assert;

import static org.mockito.ArgumentMatchers.any;

/**
 * @author akirakozov
 */
@ContextConfiguration(classes = {
        MpfsClientContextConfiguration.class,
        MulcaClientContextConfiguration.class,
        MpfsMockConfiguration.class,
        MpfsClientTest.Config.class
})
public class MpfsClientTest extends AbstractTest {
    private static final Logger logger = LoggerFactory.getLogger(MpfsClientTest.class);

    @Configuration
    public static class Config {

        @Bean
        public HttpRecorderRule httpRecorderRule() {
            return HttpRecorderRule.builder().prefix(System.getProperty("user.home") + "/arcadia/disk/support/common/src/test/resources/").build();
        }

    }

    // Album of user TestUser.uid2
    public static final String ALBUM_PUBLIC_KEY =
            "hrqswxCKCpMch9dN0L2lwjo/Ed1+yxRKWUhOsihA5uNgvahc2S7Hz8Eugb6VQ3rwq/J6bpmRyOJonT3VoXnDag==";

    @Rule
    @Autowired
    public final HttpRecorderRule recorderRule = null;

    @Autowired
    private MulcaClient mulcaClient;

    @Autowired
    private MpfsClientImpl mpfsClient;

    @Test
    @HttpRecorderRule.IgnoreParams
    public void uploadAndDownload() {
        byte[] data = "hello, mulco".getBytes();
        MulcaId mulcaId = mulcaClient.upload(InputStreamSourceUtils.bytes(data), "tmp");

        try {
            MpfsCallbackResponse response = mpfsClient.markMulcaIdToRemove(mulcaId);
            Assert.equals(response.getStatusCode(), HttpStatus.SC_OK);
        } finally {
            mulcaClient.delete(mulcaId);
        }
    }

    @Test
    public void generateZaberunUrl() {
        String url = mpfsClient.generateZaberunUrl(
                "wtf",
                "abc.txt",
                "preview",
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty());
        Check.isTrue(url.startsWith("https://downloader.dst.yandex.ru/preview/"));
        Check.isTrue(url.endsWith("?uid=0&filename=abc.txt&disposition=attachment&hash=&limit=0&content_type=application%2Foctet-stream&tknv=v2"));
    }

    @Test
    public void kladunDownloadCounterIncUri() {
        String url = mpfsClient.getKladunDownloadCounterIncUri("abc123def", 456L);
        Assert.assertContains(url, "/service/kladun_download_counter_inc?hash=abc123def&bytes=456");
    }

    @Test
    public void fileInfoByHid() {
        MpfsFileInfo fileInfo = mpfsClient.getFileInfoByHid(TestUser.TEST_HID);

        logger.info("file info: {}", fileInfo);
        Assert.some("e9c9e1110bac33b3332245f591459772", fileInfo.getMeta().getMd5());
        Assert.some("f25306361cd97508bf80f7567267a627efd3f683738ad07cf470c461b2ba7a79", fileInfo.getMeta().getSha256());
    }

    @Test
    public void fileInfoByUidAndPathWithEmptyMeta() {
        MpfsFileInfo fileInfo = mpfsClient.getFileInfoByUidAndPath(MpfsUser.of(TestUser.uid), "/disk/hello.txt", Cf.list());
        logger.info("file info: {}", fileInfo);
        Assert.some("381383ebc1c5b46bdcaca425f6e9b87a", fileInfo.getMeta().getMd5());
        Assert.some("a6ab876b7a5f1542bed5f4191f5ba366aaec77bd808e2d5b80c4195909f65900", fileInfo.getMeta().getSha256());
        Assert.some("1000004.yadisk:201515217.2496900563192734855116040009415", fileInfo.getMeta().getFileMid());
        Assert.some("bfae4387258001c13bce460304f6da77", fileInfo.getMeta().getHid());
        Assert.some("document", fileInfo.getMeta().getMediaType());
        Assert.some("text/plain", fileInfo.getMeta().getMimeType());
        Assert.equals(39L, fileInfo.getMeta().getSize().get().toBytes());
        Assert.assertContains(fileInfo.getMeta().getFileUrl().get(), "https://downloader.dst.yandex.net");
    }

    @Test
    public void fileInfoByUidAndPathWithNonEmptyMeta() {
        MpfsFileInfo fileInfo = mpfsClient.getFileInfoByUidAndPath(MpfsUser.of(TestUser.uid), "/disk/hello.txt", Cf.list("mediatype", "mimetype"));
        logger.info("file info: {}", fileInfo);
        Assert.isEmpty(fileInfo.getMeta().getMd5());
        Assert.isEmpty(fileInfo.getMeta().getSha256());
        Assert.isEmpty(fileInfo.getMeta().getFileMid());
        Assert.isEmpty(fileInfo.getMeta().getHid());
        Assert.some("document", fileInfo.getMeta().getMediaType());
        Assert.some("text/plain", fileInfo.getMeta().getMimeType());
        Assert.isEmpty(fileInfo.getMeta().getSize());
        Assert.isEmpty(fileInfo.getMeta().getFileUrl());
    }

    @Test
    public void fileInfoByUidAndPathWithNonEmptyAndDjfsUnsupportedMeta() {
        MpfsFileInfo fileInfo = mpfsClient.getFileInfoByUidAndPath(MpfsUser.of(TestUser.uid), "/disk/hello.txt", Cf.list("mediatype", "mimetype", "unsupported"));
        logger.info("file info: {}", fileInfo);
        Assert.isEmpty(fileInfo.getMeta().getMd5());
        Assert.isEmpty(fileInfo.getMeta().getSha256());
        Assert.isEmpty(fileInfo.getMeta().getFileMid());
        Assert.isEmpty(fileInfo.getMeta().getHid());
        Assert.some("document", fileInfo.getMeta().getMediaType());
        Assert.some("text/plain", fileInfo.getMeta().getMimeType());
        Assert.isEmpty(fileInfo.getMeta().getSize());
        Assert.isEmpty(fileInfo.getMeta().getFileUrl());
    }

    @Test
    public void fileInfoByUidAndPathWithNonEmptyMetaAndDjfsUnsupportedParams() {
        MpfsFileInfo fileInfo = mpfsClient.getFileInfoByUidAndPath(MpfsUser.of(TestUser.uid), "/disk/hello.txt", Cf.map("unsupported", "value"), Cf.list("mediatype", "mimetype"));
        logger.info("file info: {}", fileInfo);
        Assert.isEmpty(fileInfo.getMeta().getMd5());
        Assert.isEmpty(fileInfo.getMeta().getSha256());
        Assert.isEmpty(fileInfo.getMeta().getFileMid());
        Assert.isEmpty(fileInfo.getMeta().getHid());
        Assert.some("document", fileInfo.getMeta().getMediaType());
        Assert.some("text/plain", fileInfo.getMeta().getMimeType());
        Assert.isEmpty(fileInfo.getMeta().getSize());
        Assert.isEmpty(fileInfo.getMeta().getFileUrl());
    }

    @Test
    public void fileInfoByUidAndPathWithNonEmptyMetaAndUnsupportedPath() {
        MpfsFileInfo fileInfo = mpfsClient.getFileInfoByUidAndPath(MpfsUser.of(TestUser.uid), "/settings/hello.txt", Cf.list("mediatype", "mimetype"));
        logger.info("file info: {}", fileInfo);
        Assert.isEmpty(fileInfo.getMeta().getMd5());
        Assert.isEmpty(fileInfo.getMeta().getSha256());
        Assert.isEmpty(fileInfo.getMeta().getFileMid());
        Assert.isEmpty(fileInfo.getMeta().getHid());
        Assert.some("document", fileInfo.getMeta().getMediaType());
        Assert.some("text/plain", fileInfo.getMeta().getMimeType());
        Assert.isEmpty(fileInfo.getMeta().getSize());
        Assert.isEmpty(fileInfo.getMeta().getFileUrl());
    }

    @Test
    public void fileInfoByFileId() {
        String fileId =
                mpfsClient.getFileInfoByUidAndPath(MpfsUser.of(TestUser.uid), "/disk/hello.txt", Cf.list()).getMeta().getFileId()
                        .get();

        Assert.some(mpfsClient.getFileInfoByFileId(MpfsUser.of(TestUser.uid), fileId).getMeta().getFileId());

        Assert.assertThrows(
                () -> mpfsClient.getFileInfoByFileId(MpfsUser.of(127000), fileId),
                UserNotInitializedException.class);

    }

    @Test
    public void getAlbumResourceListByOwner() {
        MpfsCallbackResponse response = mpfsClient.getAlbumResources(ALBUM_PUBLIC_KEY, Option.of(MpfsUser.of(TestUser.uid2)));
        checkAlbumResourcesListResponse(response);
    }

    @Test
    public void testParse() {
        String userInfo = "{\"unlimited_autoupload_enabled\": 1, \"is_mailish\": 0, \"reg_time\": 1361771406, \"trash_autoclean_period\": 30, \"settings\": {\"verstka\": {\"timestampLastDisplayedDialogWelcome\": \"1400413919613\", \"countDisplaysDialogWelcome\": \"3\", \"timestampVisitPhotoslice\": \"1443345656089\", \"timestampDialogPromoOfficeClosed\": \"1447858233951\", \"showAgreementOfficeConvertation\": \"0\", \"timestampVisitFeed\": \"1486634026675\", \"timestampRecentFilesClosed\": \"0\", \"timestampVisitAlbums\": \"1464004178505\", \"timestampDropzoneClosed\": \"1439320307192\", \"spaceGivenAway\": \"0\", \"lastContext\": \"/feed\", \"timestampVisitJournal\": \"1468048497059\", \"dropzoneState\": \"closed\", \"view\": \"list\"}, \"photostream\": {\"null\": \"<><autoupload>wifi</autoupload></>\", \"7666d9a94c90f1ddbd6a303c75765608\": \"<><autoupload>wifi</autoupload></>\", \"b7bcd320d9e87c53dc73abed4c23c311\": \"<><autoupload>wifi</autoupload></>\"}}, \"locale\": \"ru\", \"space\": {\"used\": 8097856565, \"uid\": \"142581702\", \"filesize_limit\": 53687091200, \"paid_filesize_limit\": 103687091200, \"free\": 312950948811, \"limit\": 321048805376, \"trash\": 9068046, \"files_count\": 14}, \"db\": {\"status\": \"RW\", \"shard\": \"pg\"}, \"devices\": {\"mobile\": {\"7666d9a94c90f1ddbd6a303c75765608\": {\"product\": \"WW_a501cg\", \"reg_time\": 1499798649, \"install\": \"7666d9a94c90f1ddbd6a303c75765608\", \"serial\": \"EBAZFG159520\", \"os_type\": \"android\", \"manufacturer\": \"asus\"}, \"b7bcd320d9e87c53dc73abed4c23c311\": {\"product\": \"capricorn\", \"reg_time\": 1511530264, \"install\": \"b7bcd320d9e87c53dc73abed4c23c311\", \"serial\": \"5886e921\", \"os_type\": \"android\", \"manufacturer\": \"Xiaomi\"}}, \"desktop\": {\"4e588657cd36936d47893ea34693f65a\": {\"os_type\": \"cli\", \"os\": \"lin Linux 3.10.0-123.8.1.el7.x86_64 x86_64 #1 SMP Mon Sep 22 19:06:58 UTC 2014\", \"reg_time\": 1452804771, \"install\": \"096153ae-7ff0-41ea-9bca-7352d4ae0f43\"}, \"1586d5dcd1a6276a13f21cb4200dd384\": {\"product\": \"System Product Name\", \"reg_time\": 1442959351, \"install\": \"aed44716d6a64e38b1e7b42285786572\", \"serial\": \"System Serial Number\", \"os\": \"win 6.1.7601 Service Pack 1\", \"os_type\": \"windows\", \"manufacturer\": \"ASUSTeK COMPUTER INC.\"}}}, \"is_overdrawn\": 0, \"states\": {\"global\": {\"first_file\": 1, \"desktop_installed\": 1, \"promo_shared\": 1, \"narod_migrated\": 1, \"music_subscribed\": \"1\", \"yandex_staff\": \"1\", \"file_uploaded\": 1, \"webdav_used\": \"1\", \"mobile_installed\": 1}, \"billing\": {\"market\": \"RU\"}}, \"version\": 1511975127205114, \"paid\": 0, \"has_attach\": \"1\", \"advertising_enabled\": 1, \"blocked\": 0}";

        MpfsUserInfo info = MpfsUserInfo.P.parseJson(userInfo);
        Option<Integer> unlimitedAutouploadEnabled = info.unlimitedAutouploadEnabled;
        Assert.equals(unlimitedAutouploadEnabled.get(), 1);
        Option<Long> filesize_limit = info.getSpace().filesize_limit;
        Assert.equals(filesize_limit.get(), 53687091200L);
        Option<Long> paid_filesize_limit = info.getSpace().paid_filesize_limit;
        Assert.equals(paid_filesize_limit.get(), 103687091200L);
    }

    @Test
    public void getAlbumResourceListWithoutUid() {
        MpfsCallbackResponse response = mpfsClient.getAlbumResources(ALBUM_PUBLIC_KEY, Option.empty());
        checkAlbumResourcesListResponse(response);
    }

    private void checkAlbumResourcesListResponse(MpfsCallbackResponse response) {
        ListF<MpfsResourceInfo> resources = new MpfsManager().parseFullTreeOutput(response.getResponse());
        Assert.sizeIs(3, resources);
        Assert.equals("/", resources.get(0).getPath());
        Assert.equals("dir", resources.get(0).getType());

        Assert.equals("/gus.jpg", resources.get(1).getPath());
        Assert.equals("file", resources.get(1).getType());
        Assert.some("image", resources.get(1).getMediaTypeO());
        Assert.some(DataSize.fromBytes(202578), resources.get(1).getSizeO());
        Assert.some(MulcaId.fromSerializedString("1000020.yadisk:4000475747.E1554:148685303368080229229728166625"),
                resources.get(1).getMulcaIdO());

        Assert.equals("/apples.jpg", resources.get(2).getPath());
        Assert.equals("file", resources.get(2).getType());
    }

    @Test
    public void getOfficeStoreUrl() {
        Assert.assertContains(
                mpfsClient.getOfficeStoreUri("id", "token", "123"),
                "/json/office_store?access_token=token&resource_id=id&access_token_ttl=123");
    }

    @Test
    public void getShareUidsInGroup() {
        String content = "{\"users\":["
                + "{\"status\":\"owner\", \"uid\":\"158426306\"},"
                + "{\"status\":\"approved\", \"uid\":\"97963799\"}]}";

        MpfsClientImpl client = Mockito.spy(mpfsClient);

        Mockito.doReturn(new MpfsCallbackResponse(200, "OK", content, Cf.map(), Option.empty()))
                .when(client).executeAndGetResponse(Mockito.<Option<MpfsUser>>any(), Mockito.any());

        MpfsGroupUids response = client.getShareUidsInGroupO(MpfsUser.of(1), "file_id").get();

        Assert.equals(Cf.list(158426306, 97963799).map(PassportUid::cons), response.getUids());

        Mockito.doThrow(new PermanentHttpFailureException("404", 404))
                .when(client).executeAndGetResponse(Mockito.<Option<MpfsUser>>any(), Mockito.any());

        Assert.none(client.getShareUidsInGroupO(MpfsUser.of(1), "file_id"));
    }

    @Test
    public void getLentaBlocksFileIds() {
        PassportUid uid = TestUser.uid;
        MpfsUid muid = new MpfsUid(uid);

        Assert.equals(1, mpfsClient.getLentaBlocksFileIds(
                MpfsUser.of(uid), uid + ":/disk", "document", muid, 0, Integer.MAX_VALUE, 1).get().totalCount);

        Assert.hasSize(1, mpfsClient.getLentaBlocksFileIds(
                MpfsUser.of(uid), uid + ":/disk", "document", muid, 0, Integer.MAX_VALUE, 1).get().fileIds);

        Assert.equals(0, mpfsClient.getLentaBlocksFileIds(
                MpfsUser.of(uid), uid + ":/disk", "video", muid, 0, Integer.MAX_VALUE, 0).get().totalCount);

        Assert.none(mpfsClient.getLentaBlocksFileIds(
                MpfsUser.of(uid), uid + ":/blackhole", "video", muid, 0, Integer.MAX_VALUE, 0));
    }

    @Test
    public void getUserSettings() {
        String content = "{\"settings\":{"
                + "\"verstka\":{"
                + "  \"countDisplaysDialogWelcome\":\"3\","
                + "  \"timestampVisitPhotoslice\":\"1436782803420\""
                + "},"
                + "\"comments\":{"
                + "  \"timestamp_mail_notifications_v1_unsubscribe\":\"1436782803420\""
                + "}}}";

        MpfsClientImpl client = Mockito.spy(mpfsClient);

        Mockito.doReturn(new MpfsCallbackResponse(200, "OK", content, Cf.map(), Option.empty()))
                .when(client).executeAndGetResponse(Mockito.any(), Mockito.any());

        MapF<String, MapF<String, String>> settings = client.getUserSettingsO(MpfsUser.of(1)).get();

        Assert.equals("3", settings.getTs("verstka").getTs("countDisplaysDialogWelcome"));
        Assert.equals("1436782803420", settings.getTs("comments").getTs("timestamp_mail_notifications_v1_unsubscribe"));
    }

    @Test
    public void parseNoneResourceId() {
        BenderParser<ResourceIdHolder> parser = Bender.parser(ResourceIdHolder.class);

        Assert.none(parser.parseJson("{}").id);
        Assert.none(parser.parseJson("{\"id\":\"None\"}").id);
        Assert.some(MpfsResourceId.parse("5:0"), parser.parseJson("{\"id\":\"5:0\"}").id);
    }

    @Test
    public void getSharedFolderPathByUidAndGroupId() {
        Option<String> path = mpfsClient.getSharedFolderPathByUidAndGroupId(
                MpfsUser.of(TestUser.uid2), "c3988441a169867269e7a368940c4e13");
        Assert.some("/disk/shared2", path);

        path = mpfsClient.getSharedFolderPathByUidAndGroupId(MpfsUser.of(TestUser.uid2), "non_existing_group_id");
        Assert.none(path);
    }

    @Test
    public void getUserServiceInfoList() {
        // TODO: add some services to test user and make more interesting test
        ListF<MpfsUserServiceInfo> serviceInfos = mpfsClient.getUserServiceInfoList(MpfsUser.of(TestUser.uid));
        Assert.hasSize(0, serviceInfos);
    }

    @Test
    public void getFullRelativeTreePublicUri() {
        Assert.equals(
                "/json/public_fulltree?private_hash=hash_11234&meta=file_mid%2Csize%2Cmimetype%2Chid%2Cmediatype",
                Uri2.parse(mpfsClient.getFullRelativeTreePublicUri("hash_11234", false))
                        .onlyPathQueryFragment().toString());

        Assert.equals(
                "/json/public_fulltree?private_hash=hash_11234" +
                        "&meta=file_mid%2Csize%2Cmimetype%2Chid%2Cmediatype&show_nda=kladun",
                Uri2.parse(mpfsClient.getFullRelativeTreePublicUri("hash_11234", true))
                        .onlyPathQueryFragment().toString());
    }

    @Test
    public void bulkInfo() {
        ListF<MpfsFileInfo> infos = mpfsClient
                .bulkInfoByPaths(MpfsUser.of(TestUser.uid), Cf.set(),
                        Cf.list("/disk", "/disk/hello.txt", "/disk/Музыка"));

        Assert.sizeIs(3, infos);
        Assert.some("/disk", infos.get(0).path);
        Assert.some("/disk/hello.txt", infos.get(1).path);
        Assert.some("/disk/Музыка", infos.get(2).path);
    }

    @Test
    public void bulkInfoToDjfsWithMeta() {
        ListF<MpfsFileInfo> infos = mpfsClient
                .bulkInfoByPaths(MpfsUser.of(TestUser.uid), Cf.set("md5"),
                        Cf.list("/disk", "/disk/hello.txt", "/disk/Музыка"));

        Assert.sizeIs(3, infos);
        Assert.some("/disk", infos.get(0).path);
        Assert.some("/disk/hello.txt", infos.get(1).path);
        Assert.some("/disk/Музыка", infos.get(2).path);
    }

    @Test
    public void bulkInfoToMpfsWithMeta() {
        ListF<MpfsFileInfo> infos = mpfsClient
                .bulkInfoByPaths(MpfsUser.of(TestUser.uid), Cf.set("unsupported"),
                        Cf.list("/disk", "/disk/hello.txt", "/disk/Музыка"));

        Assert.sizeIs(3, infos);
        Assert.some("/disk", infos.get(0).path);
        Assert.some("/disk/hello.txt", infos.get(1).path);
        Assert.some("/disk/Музыка", infos.get(2).path);
    }

    @Test
    public void bulkInfoToMpfsWithMetaAndUnsupportedPaths() {
        ListF<MpfsFileInfo> infos = mpfsClient
                .bulkInfoByPaths(MpfsUser.of(TestUser.uid), Cf.set("md5"),
                        Cf.list("/disk", "/disk/hello.txt", "/settings/Музыка"));

        Assert.sizeIs(3, infos);
        Assert.some("/disk", infos.get(0).path);
        Assert.some("/disk/hello.txt", infos.get(1).path);
        Assert.some("/settings/Музыка", infos.get(2).path);
    }

    @Test
    public void streamingList() {
        mpfsClient.streamingListByUidAndPath(MpfsUser.of(TestUser.uid), "/disk",
                Option.empty(), Option.empty(), Option.empty(), false, it -> {
                    SetF<String> paths = Cf.x(it).map(info -> info.path.get()).toList().unique();
                    Assert.equals(Cf.set("/disk", "/disk/hello.txt", "/disk/Музыка"), paths);
                });
    }

    @BenderBindAllFields
    public static class ResourceIdHolder {
        public Option<MpfsResourceId> id;
    }

    @Test
    public void mpfsLentaParserTest() throws Exception {

        MpfsClientImpl client = new MpfsClientImpl("");
        String json = "["
                + "   {"
                + "      \"ctime\":1333569600,"
                + "      \"meta\":{"
                + "         \"file_id\":\"a5161dbc1d2704565a44d84413b71bbed65964c0c9a82e481be255cdfc3bf3d8\","
                + "         \"total_results_count\":138,"
                + "         \"file_url\":\"http_some\","
                + "         \"resource_id\":\"567060404:a5161dbc1d2704565a44d84413b71bbed65964c0c9a82e481be255cdfc3bf3d8\""
                + "      },"
                + "      \"mtime\":1333569600,"
                + "      \"path\":\"/photounlim\","
                + "      \"utime\":0,"
                + "      \"type\":\"dir\","
                + "      \"id\":\"/photounlim/\","
                + "      \"name\":\"photounlim\""
                + "   },"
                + "   {"
                + "      \"ctime\":1511176949,"
                + "      \"etime\":1480349629,"
                + "      \"meta\":{"
                + "         \"etime\":1480349629,"
                + "         \"file_id\":\"90c7556626c1b58a7f854f266d347162f4e5bad87ac9571f2253707f389d6875\","
                + "         \"resource_id\":\"567060404:90c7556626c1b58a7f854f266d347162f4e5bad87ac9571f2253707f389d6875\""
                + "      },"
                + "      \"mtime\":1511176949,"
                + "      \"path\":\"/photounlim/2016-11-28 19-13-49.JPG\","
                + "      \"utime\":1511176949,"
                + "      \"type\":\"file\","
                + "      \"id\":\"/photounlim/2016-11-28 19-13-49.JPG\","
                + "      \"name\":\"2016-11-28 19-13-49.JPG\""
                + "   }"
                + "]";
        JsonNode node = new ObjectMapper().readTree(json);
        MpfsLentaBlockFullDescription mpfsLentaBlockFullDescription = client.buildresultMpfsLenta(node, null);

        MpfsLentaBlockItemDescription lentaBlock = mpfsLentaBlockFullDescription.lentaBlock;
        Assert.assertTrue(lentaBlock.url.isPresent());
        Assert.equals(Option.of("http_some"), lentaBlock.url);
        Assert.notNull(lentaBlock.id);
        Assert.equals("/photounlim/", lentaBlock.id);
        Assert.notNull(lentaBlock.resourceId);
        Assert.equals("567060404:a5161dbc1d2704565a44d84413b71bbed65964c0c9a82e481be255cdfc3bf3d8", lentaBlock.resourceId);
        Assert.notNull(lentaBlock.fileId);
        Assert.equals("a5161dbc1d2704565a44d84413b71bbed65964c0c9a82e481be255cdfc3bf3d8", lentaBlock.fileId);

    }

    @Test
    public void isQuickMoveEnabledTrue() {
        Assert.isTrue(mpfsClient.isQuickMoveEnabled("4020167460"));
    }

    @Test
    public void isQuickMoveEnabledFalse() {
        Assert.isFalse(mpfsClient.isQuickMoveEnabled("4020085842"));
    }

    @Test
    public void defaultFolders() {
        MapF<String, String> result = mpfsClient.getDefaultFolders(MpfsUser.of(TestUser.uid));
        Assert.isTrue(result.containsKeyTs("attach"));
        Assert.assertEquals("/disk/Почтовые вложения", result.getTs("attach"));
    }

    @Test(expected = UserNotInitializedException.class)
    public void testFeatureToggles400() throws IOException {
        testIsDiskProEnableWithHttpError(400, MpfsRequestExecutor.USER_NOT_INITIALIZED_CODE);
    }

    @Test(expected = UserBlockedException.class)
    public void testFeatureToggles403() throws IOException {
        testIsDiskProEnableWithHttpError(403, MpfsRequestExecutor.USER_BLOCKED_CODE);
    }

    @Test(expected = PermanentHttpFailureException.class)
    public void testFeatureToggles404() throws IOException {
        testIsDiskProEnableWithHttpError(404, "404");
    }

    private void testIsDiskProEnableWithHttpError(int statusCode, String mpfsStatusCode) throws IOException {
        HttpClient httpClient = Mockito.mock(HttpClient.class);
        MpfsClient mpfsClient = new MpfsClientImpl(httpClient, httpClient, "", MpfsUserInitParamsExtractor.empty(),
                false, Option.empty(), Option.empty(), false);

        PassportUid uid = PassportUid.cons(1);
        Mockito.when(httpClient.execute(any(HttpUriRequest.class), any(), any()))
                .thenReturn(new MpfsCallbackResponse(statusCode, "", "{\"code\":" + mpfsStatusCode + "}", Cf.map(),
                        Option.empty()));

        mpfsClient.getFeatureToggles(MpfsUser.of(uid));
    }

}
