package ru.yandex.chemodan.mpfs;

import java.io.Closeable;
import java.io.IOException;
import java.net.URI;

import com.fasterxml.jackson.databind.JsonNode;
import org.apache.http.HttpMessage;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class MpfsRequestExecutor implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(MpfsRequestExecutor.class);

    public static final String USER_NOT_INITIALIZED_CODE = "44";
    public static final String USER_BLOCKED_CODE = "119";

    private final HttpClient httpClient;
    private final HttpClient longRequestsHttpClient;
    private final Option<Function1V<MpfsUser>> reinitFn;

    public MpfsRequestExecutor(HttpClient httpClient, HttpClient longRequestsHttpClient,
            Option<Function1V<MpfsUser>> reinitFn)
    {
        this.httpClient = httpClient;
        this.longRequestsHttpClient = longRequestsHttpClient;
        this.reinitFn = reinitFn;
    }

    public MpfsCallbackResponse executeAndGetResponse(Option<MpfsUser> uid, String uriStr) {
        return executeAndGetResponse(uid, uriStr, false, Tuple2List.tuple2List());
    }

    public MpfsCallbackResponse executeAndGetResponse(Option<MpfsUser> uid, String uriStr,
            Tuple2List<String, String> headers)
    {
        return executeAndGetResponse(uid, uriStr, false, headers);
    }

    public MpfsCallbackResponse executeLongRequestAndGetResponse(Option<MpfsUser> uid, String uriStr) {
        return executeAndGetResponse(uid, uriStr, true, Tuple2List.tuple2List());
    }

    private MpfsCallbackResponse executeAndGetResponse(Option<MpfsUser> uid, String uriStr, boolean isLongRequest,
            Tuple2List<String, String> headers)
    {
        HttpClient clientToUse = isLongRequest ? longRequestsHttpClient : httpClient;
        URI uri = URI.create(uriStr);

        HttpGet get = withYcridHeaderSet(new HttpGet(uri));
        headers.forEach((Function2V<String, String>) get::addHeader);

        MpfsCallbackResponse response =
                ApacheHttpClientUtils.execute(get, clientToUse, new MpfsCallbackResponseHandler(uri));

        return processMpfsResponseStatusCode(uid, clientToUse, uri, response);
    }

    private MpfsCallbackResponse processMpfsResponseStatusCode(Option<MpfsUser> uid, HttpClient clientToUse, URI uri,
            MpfsCallbackResponse response)
    {
        int statusCode = response.getStatusCode();
        if (statusCode == 400 && isErrorCodeEquals(response, USER_NOT_INITIALIZED_CODE)) {
            if (!reinitFn.isPresent()) {
                // Во всех приложенях кроме вебдава будет выбрасываться это исключение и можно на него реагировать
                throw new UserNotInitializedException(uid.getOrElse(MpfsUser.of(0)));
            }

            // В вебдаве будем пробовать инициализировать юзера и если это не получилось - прокидывать наружу исходных ответ
            // Потому что при получении кода 409 от UserNotInitializedException клиенты начинают ретраить
            try {
                logger.info("Reinitializing user {} due to bad mpfs response",
                        uid.getOrThrow("Missing user to initialize").getUidStr());

                reinitFn.getOrThrow("no reinitFn present").apply(uid.get());

                response = ApacheHttpClientUtils.execute(
                        withYcridHeaderSet(new HttpGet(uri)), clientToUse, new MpfsCallbackResponseHandler(uri));
                statusCode = response.getStatusCode();
            } catch (PermanentHttpFailureException e) {
                Level level = Level.ERROR;
                // https://st.yandex-team.ru/CHEMODAN-66706
                if (e.getMessage().contains("account has no password")) {
                    level = Level.DEBUG;
                }
                logger.log(level, "Failed to init user: {}", e);
            } catch (RuntimeException e) {
                logger.error("Failed to init user: {}", e);
            }
        }

        if (statusCode == 403 && isErrorCodeEquals(response, USER_BLOCKED_CODE)) {
            throw new UserBlockedException(uid.getOrElse(MpfsUser.of(0)));
        }

        if (HttpStatus.is2xx(statusCode)) {
            return response;
        } else {
            if (HttpStatus.is4xx(statusCode)) {
                throw new PermanentHttpFailureException(response.getResponse(), response.toString(), statusCode,
                        response.getHeaders());
            } else {
                throw new HttpException(statusCode, response.getResponse(), response.toString());
            }
        }
    }

    public static boolean isErrorCodeEquals(MpfsCallbackResponse response, String code) {
        try {
            JsonNode codeNode = JsonNodeUtils.getNode(response.getResponse()).path("code");
            if (!codeNode.isMissingNode() && code.equals(codeNode.asText())) {
                return true;
            }
        } catch (RuntimeException e) {
            logger.error("Failed to parse error response body: {}", e);
        }
        return false;
    }

    public <T> T executeGetWithReinit(Option<MpfsUser> uid, String uriStr, ResponseHandler<T> responseHandler) {
        return executeGet(uriStr, (response) -> {
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode == 400) {
                try {
                    String responseText = InputStreamSourceUtils.wrap(response.getEntity().getContent())
                            .limit(DataSize.fromKiloBytes(10).toBytes())
                            .readText();

                    JsonNode code = JsonNodeUtils.getNode(responseText).path("code");
                    if (!code.isMissingNode() && USER_NOT_INITIALIZED_CODE.equals(code.asText())) {
                        if (reinitFn.isPresent()) {
                            logger.info("Reinitializing user {} due to bad mpfs response",
                                    uid.getOrThrow("Missing user to initialize").getUidStr());

                            reinitFn.get().apply(uid.get());
                        } else {
                            throw new UserNotInitializedException(uid.get());
                        }
                    } else {
                        throw new PermanentHttpFailureException(responseText, statusCode);
                    }
                } catch (Exception e) {
                    logger.error("Failed to parse response body for 400: {}", e);
                    throw ExceptionUtils.translate(e);
                }

                return executeGet(uriStr, responseHandler);
            } else {
                return responseHandler.handleResponse(response);
            }
        });
    }

    public <T> T executeGet(String uri, ResponseHandler<T> responseHandler) {
        return ApacheHttpClientUtils.execute(withYcridHeaderSet(new HttpGet(uri)), httpClient, responseHandler);
    }

    public MpfsCallbackResponse postAndGetResponse(Option<MpfsUser> uid, String uriStr, byte[] content) {
        final MpfsCallbackResponse response = postAndGetResponse(uriStr, content);
        URI uri = URI.create(uriStr);
        return processMpfsResponseStatusCode(uid, httpClient, uri, response);
    }

    public MpfsCallbackResponse postAndGetResponse(String uriStr, byte[] content) {
        return postAndGetResponse(uriStr, content, Tuple2List.fromPairs());
    }

    public MpfsCallbackResponse executeAndGetResponse(Option<MpfsUser> uid, String uriStr,Tuple2List<String, String> headers, byte[] content) {
        URI uri = URI.create(uriStr);

        HttpGetWithBody put = withYcridHeaderSet(new HttpGetWithBody(uriStr));
        headers.forEach((Function2V<String, String>) put::addHeader);
        put.setEntity(new ByteArrayEntity(content));
        MpfsCallbackResponse response =
                ApacheHttpClientUtils.execute(put, httpClient, new MpfsCallbackResponseHandler(uri));
        return processMpfsResponseStatusCode(uid,httpClient,uri,response);
    }

    public MpfsCallbackResponse postAndGetResponse(String uriStr, byte[] content, Tuple2List<String, String> headers) {
        URI uri = URI.create(uriStr);

        HttpPost post = withYcridHeaderSet(new HttpPost(uri));
        headers.forEach((Function2V<String, String>) post::addHeader);
        post.setEntity(new ByteArrayEntity(content));
        return ApacheHttpClientUtils.execute(post, httpClient, new MpfsCallbackResponseHandler(uri));
    }

    public static <T extends HttpMessage> T withYcridHeaderSet(T request) {
        YandexCloudRequestIdHolder.getO().forEach(rid -> request.addHeader(CommonHeaders.YANDEX_CLOUD_REQUEST_ID, rid));
        return request;
    }

    @Override
    public void close() throws IOException {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    private static class HttpGetWithBody extends HttpEntityEnclosingRequestBase {

        public HttpGetWithBody(String uri) {
            super();
            setURI(URI.create(uri));
        }

        @Override
        public String getMethod() {
            return HttpGet.METHOD_NAME;
        }
    }
}
