package ru.yandex.chemodan.mpfs;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTimeConstants;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author ssytnik
 */
@RequiredArgsConstructor
public class MpfsResourceInfo extends DefaultObject {
    private final Instant ctime;
    private final Instant mtime;
    private final String path;
    private final String type;
    private final String id;
    private final String name;
    private final MapF<String, Object> meta;

    public static MpfsResourceInfo cons(JsonNode o) {
        MapF<String, Object> meta = Cf.hashMap();

        Option.ofNullable(o.get("meta")).ifPresent(metaNode -> metaNode.fields().forEachRemaining(e -> {
            if (e.getValue().isTextual()) {
                meta.put(e.getKey(), e.getValue().textValue());
            } else if (e.getValue().isNumber()) {
                meta.put(e.getKey(), e.getValue().longValue());
            } else {
                throw new RuntimeException("Unsupported meta value: " + e.getValue());
            }
        }));

        return new MpfsResourceInfo(
                new Instant( o.get("ctime").longValue() * DateTimeConstants.MILLIS_PER_SECOND ),
                new Instant( o.get("mtime").longValue() * DateTimeConstants.MILLIS_PER_SECOND ),
                o.get("path").textValue(),
                o.get("type").textValue(),
                o.get("id").textValue(),
                o.get("name").textValue(),
                meta);
    }

    public Instant getCtime() {
        return ctime;
    }

    public Instant getMtime() {
        return mtime;
    }

    public String getPath() {
        return path;
    }

    public String getNormalizedPath() {
        return isFolder() && !path.endsWith("/") ? path + "/" : path;
    }

    // note: this works with relative paths only (see mpfsClient)
    public boolean isRelativePathRoot() {
        return "/".equals(getNormalizedPath());
    }

    public String getType() {
        return type;
    }

    public boolean isFolder() {
        return "dir".equals(type);
    }

    public String getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public MapF<String, Object> getMeta() {
        return meta;
    }

    public Option<MulcaId> getMulcaIdO() {
        return meta.getO("file_mid").<String>uncheckedCast().map(MulcaId::fromSerializedString);
    }

    public Option<DataSize> getSizeO() {
        return meta.getO("size").<Long>uncheckedCast().map(DataSize::fromBytes);
    }

    public Option<String> getMimeTypeO() {
        return meta.getO("mimetype").uncheckedCast();
    }

    public Option<String> getHidO() {
        return meta.getO("hid").uncheckedCast();
    }

    public Option<String> getMediaTypeO() {
        return meta.getO("mediatype").uncheckedCast();
    }

    public Option<String> getUidO() {
        return meta.getO("uid").uncheckedCast();
    }

    public static Function<MpfsResourceInfo, Option<Long>> getSizeOF() {
        return a -> a.getSizeO().map(DataSize::toBytes);
    }

}
