package ru.yandex.chemodan.mpfs;

import java.util.Set;

import com.fasterxml.jackson.databind.JsonNode;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author friendlyevil
 */

public final class MpfsResponseParserUtils {
    private static final Logger log = LoggerFactory.getLogger(MpfsResponseParserUtils.class);

    public static Option<String> getTitleFromMpfsResponse(Option<String> responseBody, Set<Integer> whiteListCodes) {
        return getTitleFromMpfsResponse(Option.empty(), responseBody, whiteListCodes, log);
    }

    public static Option<String> getTitleFromMpfsResponse(int httpCode, Option<String> responseBody,
                                                          Set<Integer> allowedMpfsStatusCodes) {
        return getTitleFromMpfsResponse(httpCode, responseBody, allowedMpfsStatusCodes, log);
    }

    public static Option<String> getTitleFromMpfsResponse(int httpCode, Option<String> responseBody,
                                                          Set<Integer> allowedMpfsStatusCodes, Logger logger) {
        return getTitleFromMpfsResponse(Option.of(httpCode), responseBody, allowedMpfsStatusCodes, logger);
    }

    public static Option<String> getTitleFromMpfsResponse(Option<Integer> httpCode, Option<String> responseBody,
                                                          Set<Integer> allowedMpfsStatusCodes, Logger logger) {
        try {
            Option<JsonNode> mpfsResponse = parseMpfsResponse(responseBody, logger);
            if (mpfsResponse.isEmpty()) {
                return Option.empty();
            }

            Option<Integer> code = getTextField(mpfsResponse.get(),"code").flatMapO(Cf.Integer::parseSafe);

            if ((httpCode.isPresent() && HttpStatus.is5xx(httpCode.get())) ||
                    (code.isPresent() && allowedMpfsStatusCodes.contains(code.get()))) {
                return getTextField(mpfsResponse.get(), "title");
            }

            return Option.empty();
        } catch (RuntimeException e) {
            logger.error("Failed to parse mpfs response: {}, {}", responseBody, e);
            return Option.empty();
        }
    }

    public static Option<String> getTextField(JsonNode mpfsResponse, String field) {
        return getFieldAndProcess(mpfsResponse, field, JsonNode::asText);
    }

    public static <R> Option<R> getFieldAndProcess(JsonNode mpfsResponse, String field, Function<JsonNode, R> process) {
        return Option.ofNullable(mpfsResponse.get(field)).map(process);
    }

    public static Option<JsonNode> parseMpfsResponse(Option<String> responseBody, Logger logger) {
        try {
            return Option.ofNullable(JsonNodeUtils.getNode(responseBody.orElse("{}")));
        } catch (RuntimeException e) {
            logger.error("Failed to parse mpfs response: {}, {}", responseBody, e);
            return Option.empty();
        }
    }
}
