package ru.yandex.chemodan.orchestrator;

import org.apache.http.client.config.RequestConfig;
import org.junit.FixMethodOrder;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runners.MethodSorters;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.chemodan.util.test.HttpRecorderRule;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.test.Assert;

@FixMethodOrder(MethodSorters.JVM)
@ContextConfiguration(classes = OrchestratorClientTest.Context.class)
public class OrchestratorClientTest extends AbstractTest {

    private static final String GROUP_ID = "groupId";
    private static final String EXISTING_SESSION_ID = "sessionId";
    private static final String MISSING_SESSION_ID = "missingSessionId";

    @Autowired
    private OrchestratorClient orchestratorClient;

    @Rule
    @Autowired
    public final HttpRecorderRule recorderRule = null;

    @Test
    public void testCreateSession() {
        OrchestratorSession session = orchestratorClient.createSession(EXISTING_SESSION_ID, GROUP_ID);
        Assert.isTrue(session.getContainer().getHost().format().endsWith("yp-c.yandex.net"));
    }

    @Test
    public void testGetExistingSession() {
        Option<OrchestratorSession> session = orchestratorClient.getSession(EXISTING_SESSION_ID);
        Assert.some(GROUP_ID, session.map(OrchestratorSession::getGroupId));
    }

    @Test
    public void testGetMissingSession() {
        Assert.none(orchestratorClient.getSession(MISSING_SESSION_ID));
    }

    @Test
    public void testRefreshExistingSession() {
        Assert.isTrue(orchestratorClient.refreshSession(EXISTING_SESSION_ID));
    }

    @Test
    public void testRefreshMissingSession() {
        Assert.isFalse(orchestratorClient.refreshSession(MISSING_SESSION_ID));
    }

    @Test
    public void testFinishSession() {
        orchestratorClient.finishSession(EXISTING_SESSION_ID);
    }

    @Configuration
    @Import(OrchestratorClientContextConfiguration.class)
    public static class Context {

        @Bean
        public HttpRecorderRule httpRecorderRule() {
            return HttpRecorderRule.builder().prefix(System.getProperty("user.home") +
                    "/arc/arcadia/disk/support/common/src/test/resources/").build();
        }

        @Bean
        @Qualifier("orchestrator")
        @OverridableValuePrefix("orchestrator")
        public HttpClientConfigurator orchestratorHttpClientConfigurator(HttpRecorderRule rule) {
            return new HttpClientConfigurator() {
                public ApacheHttpClientUtils.Builder createBuilder() {
                    return super.createBuilder()
                            .withRequestConfig(RequestConfig.custom().setProxy(rule.getHost()).build())
                            .withRoutePlanner(new HttpRecorderRule.CustomHttpsProxyRoutePlanner())
                            .withRequestRetry();
                }
            };
        }
    }
}
