package ru.yandex.chemodan.ping;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @param <T> ping result type
 * @author nshmakov
 */
public class HostsManager<T> extends DelayingWorkerServiceBeanSupport {

    private static final Logger logger = LoggerFactory.getLogger(HostsManager.class);

    private PingClient<T> pingClient;
    private ExecutorService pingPool;
    private HostsProvider hostsProvider;

    private volatile MapF<String, T> lastPingResult = Cf.map();

    public HostsManager(HostsProvider hostsProvider, PingClient<T> pingClient, ExecutorService pingPool) {
        this.hostsProvider = hostsProvider;
        this.pingPool = pingPool;
        this.pingClient = pingClient;
    }

    public MapF<String, T> getAvailableHostsResults() {
        return lastPingResult;
    }

    public ListF<String> getAvailableHosts() {
        return lastPingResult.keys();
    }

    public ListF<String> getAllHosts() {
        return hostsProvider.getHosts();
    }

    public void refresh() throws Exception {
        execute();
    }

    @Override
    protected void execute() throws Exception {
        MapF<String, Future<T>> futures = submitPingTasks(hostsProvider.getHosts());
        lastPingResult = getResultFromFutures(futures);
    }

    private MapF<String, T> getResultFromFutures(
            MapF<String, Future<T>> futures)
    {
        MapF<String, T> uploadersStatus = Cf.hashMap();
        for (Tuple2<String, Future<T>> future : futures.entries()) {
            String host = future._1;
            try {
                T result = future._2.get();
                uploadersStatus.put(host, result);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return uploadersStatus;
            } catch (ExecutionException e) {
                logger.warn("Host {} is unavailable ({})", host, e.getMessage());
            }
        }
        return uploadersStatus;
    }

    private MapF<String, Future<T>> submitPingTasks(ListF<String> urls) {
        return urls.toMap(url -> Tuple2.tuple(url, pingPool.submit(pingCallback(url))));
    }

    private Callable<T> pingCallback(final String url) {
        return () -> pingClient.ping(url);
    }
}
