package ru.yandex.chemodan.ratelimiter;

import java.lang.reflect.Proxy;
import java.util.concurrent.Callable;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.util.AppNameHolder;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.alive2.AliveAppsListChangeListener;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.misc.concurrent.RpsLimiter;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.version.AppName;

/**
 * @author tolmalev
 */
public class ClusteredRpsLimiter implements AliveAppsListChangeListener {
    private static final Logger logger = LoggerFactory.getLogger(ClusteredRpsLimiter.class);

    private final String name;
    private final DynamicProperty<Integer> maxTotalLimit;

    private final RpsLimiter rpsLimiter;
    private volatile int appsCount = 0;

    public ClusteredRpsLimiter(String name, int maxTotalLimit, DynamicPropertyManager dynamicPropertyManager) {
        this.rpsLimiter = new RpsLimiter(0);
        this.name = name;
        this.maxTotalLimit = new DynamicProperty<>("cluster-rps-limiter-" + name, maxTotalLimit);
        if (dynamicPropertyManager != null) {
            dynamicPropertyManager.registerWatcher(this.maxTotalLimit, this::updateRps);
        }
    }

    @Override
    public void listChanged(ListF<AliveAppInfo> newList) {
        AppName appName = AppNameHolder.get();

        appsCount = newList.count(app -> app.getAppName().equals(appName.appName()));
        updateRps(maxTotalLimit.get());
    }

    private void updateRps(int maxTotalRps) {
        int rps = appsCount != 0
                ? maxTotalLimit.get() / appsCount
                : 0;

        logger.debug("Update rps limit for limiter '{}'. app_count={}, max_total_rps={}, rps_per_host={}", name, appsCount, maxTotalLimit.get(), rps);
        rpsLimiter.setRpsLimit(rps);
    }

    public void execute(Function0V function) {
        rpsLimiter.execute(function);
    }

    public <T> T execute(Function0<T> function) {
        return rpsLimiter.execute(function);
    }

    public <T> T executeWithException(Callable<T> function) throws Exception {
        return rpsLimiter.executeWithException(function);
    }

    public <T> T wrapLimited(T delegate, Class<T> interfaceClass) {
        return (T) Proxy.newProxyInstance(
                delegate.getClass().getClassLoader(),
                new Class[]{ interfaceClass },
                new RpsLimitedInvocationHandler<>(delegate, this)
        );
    }
}
