package ru.yandex.chemodan.ratelimiter;

import java.io.IOException;
import java.net.URI;
import java.text.DecimalFormat;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.util.EntityUtils;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Pattern2;

import static ru.yandex.misc.io.http.HttpStatus.SC_200_OK;
import static ru.yandex.misc.io.http.HttpStatus.SC_202_ACCEPTED;
import static ru.yandex.misc.io.http.HttpStatus.SC_404_NOT_FOUND;

/**
 * @author osidorkin
 * @author Dmitriy Amelin (lemeh)
 */
public class RateLimiterClient {
    static final int SC_429_TOO_MANY_REQUESTS = 429;

    private static final ListF<Integer> KNOWN_STATUS_CODES = Cf.list(
            SC_200_OK, SC_202_ACCEPTED, SC_404_NOT_FOUND, SC_429_TOO_MANY_REQUESTS
    );

    private static final Pattern2 WAIT_PATTERN = Pattern2.compile("wait_ms=(\\d+)");

    private static final DecimalFormat RPS_FORMAT = new DecimalFormat("#.####");

    private static final Logger logger = LoggerFactory.getLogger(RateLimiterClient.class);

    private final HttpClient httpClient;

    private final URI baseUri;

    private final String group;

    public RateLimiterClient(HttpClient httpClient, URI baseUri, String group, boolean clientWaits) {
        this.httpClient = httpClient;
        this.baseUri = UriBuilder.cons(baseUri)
                .appendPath(group)
                .addParam("client_waits", clientWaits)
                .build();
        this.group = group;
    }

    public RateLimit queryLimit(String id) {
        return queryRateLimiter(id, Option.empty(), Option.empty());
    }

    public RateLimit queryAdHoc(String id, long burst, double rps) {
        return queryRateLimiter(id, Option.of(burst), Option.of(rps));
    }

    private RateLimit queryRateLimiter(String id, Option<Long> burst, Option<Double> rps) {
        UriBuilder uriBuilder = UriBuilder.cons(baseUri)
                .appendPath(id);
        if (rps.isPresent()) {
            uriBuilder = uriBuilder.addParam("rps", RPS_FORMAT.format(rps.get()));
            if (burst.isPresent()) {
                uriBuilder = uriBuilder.addParam("burst", burst.get());
            }
        }
        URI uri = uriBuilder.build();

        try {
            HttpResponse response = httpClient.execute(new HttpPost(uri));
            return process(
                    response.getStatusLine().getStatusCode(),
                    Option.ofNullable(EntityUtils.toString(response.getEntity()))
            );
        } catch (RuntimeException | IOException e) {
            logger.warn("Failed to query rate limiter: {}", e);
            return RateLimit.proceedWithoutDelay();
        }
    }

    RateLimit process(int statusCode, Option<String> content) {
        logStatusCode(statusCode);

        switch (statusCode) {
            case SC_429_TOO_MANY_REQUESTS:
                return RateLimit.abort();

            case SC_202_ACCEPTED:
                if (!content.isPresent()) {
                    return RateLimit.proceedWithoutDelay();
                }

                return RateLimit.proceedWithDelay(
                        parseDelayO(content.get())
                                .getOrElse(Duration.ZERO)
                );

            default:
                return RateLimit.proceedWithoutDelay();
        }
    }

    private Option<Duration> parseDelayO(String responseBody) {
        return WAIT_PATTERN
                .findFirstGroup(responseBody)
                .map(Long::parseLong)
                .map(Duration::millis);
    }

    private void logStatusCode(int statusCode) {
        switch (statusCode) {
            case SC_404_NOT_FOUND:
                logger.warn("Rate limiter group = {} not found: check ratelimiter config", group);

            default:
                if (HttpStatus.is5xx(statusCode)) {
                    logger.warn("Rate limiter error: {}", statusCode);
                } else if (!KNOWN_STATUS_CODES.containsTs(statusCode)) {
                    logger.warn("Got unknown status code = {} from rate limiter", statusCode);
                }
        }
    }
}
