package ru.yandex.chemodan.ratelimiter;

import java.net.URI;

import org.apache.http.client.HttpClient;
import org.joda.time.Duration;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class RateLimiterClientTest {
    private final RateLimiterClient client = new RateLimiterClient(
            Mockito.mock(HttpClient.class),
            URI.create("http://localhost/"),
            "test-group",
            true);

    @Test
    public void testProcessForSc429TooManyRequests() {
        assertRateLimit(RateLimit.abort(), RateLimiterClient.SC_429_TOO_MANY_REQUESTS);
    }

    @Test
    public void testProcessForSc200Ok() {
        assertRateLimit(RateLimit.proceedWithoutDelay(), HttpStatus.SC_200_OK);
    }

    @Test
    public void testProcessForSc202Accepted() {
        assertRateLimit(RateLimit.proceedWithDelay(Duration.millis(1234)), HttpStatus.SC_202_ACCEPTED, "wait_ms=1234");
    }

    @Test
    public void testProcessForSc202AcceptedWithEmptyBody() {
        assertRateLimit(RateLimit.proceedWithoutDelay(), HttpStatus.SC_202_ACCEPTED);
    }

    @Test
    public void testProcessForSc202AcceptedWithBadBody() {
        assertRateLimit(RateLimit.proceedWithoutDelay(), HttpStatus.SC_202_ACCEPTED, "gibberish");
    }

    @Test
    public void testProcessForSc500() {
        assertRateLimit(RateLimit.proceedWithoutDelay(), HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
    }

    private void assertRateLimit(RateLimit expected, int statusCode) {
        assertRateLimit(expected, statusCode, Option.empty());
    }

    private void assertRateLimit(RateLimit expected, int statusCode, String content) {
        assertRateLimit(expected, statusCode, Option.of(content));
    }

    private void assertRateLimit(RateLimit expected, int statusCode, Option<String> content) {
        Assert.equals(expected, client.process(statusCode, content));
    }
}
