package ru.yandex.chemodan.ratelimiter.chunk;

import java.util.concurrent.atomic.AtomicLong;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.ratelimiter.NegativeRateException;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
public class ChunkRateLimiterImpl implements ChunkRateLimiter {
    private final static Long NANO_IN_MILLIS = 1000000L;

    private final AtomicLong openingTime;
    private final AtomicLong allowedExecutionTime;
    private final AtomicLong nanoBuffer;                // ns
    protected final Function0<Double> rateSupplier;       // units per ms
    private final Function0<Long> maxTimeSlotSupplier;  // ms
    private final Function0<Option<Long>> maxAwaitTimeSupplier; // ms
    private final Function0<Integer> defaultChunkSizeSupplier;

    public ChunkRateLimiterImpl(
            Function0<Double> rateSupplier,
            Function0<Long> maxTimeSlotSupplier,
            Function0<Option<Long>> maxAwaitTimeSupplier,
            Function0<Integer> defaultChunkSizeSupplier)
    {
        this.rateSupplier = rateSupplier;
        this.maxTimeSlotSupplier = maxTimeSlotSupplier;
        this.maxAwaitTimeSupplier = maxAwaitTimeSupplier;
        this.defaultChunkSizeSupplier = defaultChunkSizeSupplier;
        this.openingTime = new AtomicLong(0);
        this.allowedExecutionTime = new AtomicLong(0);
        this.nanoBuffer = new AtomicLong(0);
    }

    private long now() {
        return System.currentTimeMillis();
    }

    private void awaitAndShiftTime(AtomicLong expectedTime, int chunkSize) {
        ThreadUtils.sleepUntil(expectedTime.getAndUpdate(currentExpectedTime ->
                Math.max(now(), currentExpectedTime) + getRequestTimeSlot(chunkSize)));
    }

    @Override
    public int getDefaultChunkSize() {
        return defaultChunkSizeSupplier.apply();
    }

    @Override
    public <T> T acquirePermitAndExecute(int chunkSize, Function<Integer, T> action) {
        maxAwaitTimeSupplier.apply().ifPresent(maxAwaitTime ->
                Check.gt(now() + maxAwaitTime, openingTime.get(),
                        "Rate limiter wont't open within next " + maxAwaitTime + " ms"));
        Check.ge(chunkSize, 0, "Chunk size can not be negative");
        awaitAndShiftTime(openingTime, chunkSize);          // expected execution time
        awaitAndShiftTime(allowedExecutionTime, chunkSize); // sleep more if too little time passed since last request
        T result = action.apply(chunkSize);
        onActionDone(chunkSize);
        return result;
    }

    protected void onActionDone(int chunkSize) {
    }

    public boolean hasAwaitingRequests() {
        return now() < openingTime.get();
    }

    private long getRequestTimeSlot(int chunkSize) {
        double rate = rateSupplier.apply();
        if (rate <= 0.0) {
            throw new NegativeRateException("Database is overloaded");
        }
        double timeSlot = ((double) chunkSize / rate);
        long millisTimeSlot = (long) timeSlot;
        long nanoTail = (long) ((timeSlot - Math.floor(timeSlot)) * NANO_IN_MILLIS);
        millisTimeSlot += nanoBuffer.getAndUpdate(value -> (value % NANO_IN_MILLIS) + nanoTail) / NANO_IN_MILLIS;
        return Math.min(millisTimeSlot, maxTimeSlotSupplier.apply());
    }
}
