package ru.yandex.chemodan.ratelimiter.chunk;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;

import com.google.common.util.concurrent.AtomicDouble;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Try;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
public class ChunkRateLimiterImplTest {
    private CountDownLatch latch;
    private ListF<Instant> calls;
    private ExecutorService service;
    private AtomicDouble rateHolder;
    private AtomicLong maxTimeSlotHolder;
    private ChunkRateLimiterWithMeter rateLimiter;

    @Before
    public void setup() {
        service = Executors.newFixedThreadPool(10);
        calls = Cf.arrayList();
        rateHolder = new AtomicDouble();
        maxTimeSlotHolder = new AtomicLong(10000);
        rateLimiter = new ChunkRateLimiterWithMeter(
                rateHolder::get, maxTimeSlotHolder::get,
                () -> Option.of(10000L), () -> 1, Duration.millis(120), Duration.millis(1));
    }

    @Test
    public void constantRateTest() throws InterruptedException {
        rateHolder.set(5.0);
        latch = new CountDownLatch(6);

        runRequests(Cf.list(200, 200, 200, 400, 400, 400));
        latch.await();

        double realRate = rateLimiter.getRealRate();
        Assert.equals(6.6, realRate, 0.1); // 2 requests of 400 within last 120ms
        assertSomethingLike(Cf.list(40L, 40L, 40L, 80L, 80L), getIntervals());
    }

    @Test
    public void notUsedAnymoreLimiter() throws InterruptedException {
        rateHolder.set(5.0);
        latch = new CountDownLatch(2);

        runRequests(Cf.list(400, 400));
        latch.await();
        double rateFor2Requests = rateLimiter.getRealRate();
        ThreadUtils.sleep(100);
        double rateFor1Request = rateLimiter.getRealRate();
        ThreadUtils.sleep(80);
        double rateForNoRequest = rateLimiter.getRealRate();

        Assert.equals(6.6, rateFor2Requests, 0.1);
        Assert.equals(3.3, rateFor1Request, 0.1);
        Assert.equals(0.0, rateForNoRequest, 0.1);
    }

    @Test
    public void increaseRateTest() throws InterruptedException {
        rateHolder.set(5.0);
        latch = new CountDownLatch(6);

        runRequests(Cf.list(200, 200, 200));
        ThreadUtils.sleep(20);
        rateHolder.set(10.0);
        runRequests(Cf.list(200, 200, 200));
        latch.await();

        assertSomethingLike(Cf.list(40L, 40L, 40L, 20L, 20L), getIntervals());
    }

    @Test
    public void decreaseRateTest() throws InterruptedException {
        rateHolder.set(5.0);
        latch = new CountDownLatch(6);

        runRequests(Cf.list(200, 200, 200, 200, 200, 200));
        rateHolder.set(1.0);
        latch.await();

        assertSomethingLike(Cf.list(200L), getIntervals().lastO());
    }

    @Test
    public void maxTimeSlotTest() throws InterruptedException {
        rateHolder.set(1.0);
        maxTimeSlotHolder.set(100);
        latch = new CountDownLatch(2);

        runRequests(Cf.list(100000, 10));
        latch.await();

        assertSomethingLike(Cf.list(100L), getIntervals());
    }

    @Test
    public void maxAwaitTimeTest() {
        rateHolder.set(1.0);
        latch = new CountDownLatch(2);
        runRequests(Cf.list(8000, 8000));

        Assert.failure(Try.tryCatchException(
                () -> rateLimiter.acquirePermitAndExecute(8000, size -> calls.add(Instant.now()))),
                IllegalStateException.class);
    }

    private void runRequests(ListF<Integer> requests) {
        long pause = requests.unique().size() > 1 ? 10 : 0;
        requests.forEach(chunkSize -> {
            service.submit(() -> {
                rateLimiter.acquirePermitAndExecute(chunkSize, size -> calls.add(Instant.now()));
                latch.countDown();
            });
            ThreadUtils.sleep(pause);
        });
    }

    private ListF<Long> getIntervals() {
        return calls.rdrop(1).zip(calls.drop(1)).map((i1, i2) -> new Duration(i1, i2).getMillis());
    }

    private void assertSomethingLike(ListF<Long> expected, ListF<Long> value) {
        System.out.println(expected + " " + value);
        Assert.sameSize(expected, value);
        Assert.forAll(expected.zip(value), this::somethingLike);
    }

    private boolean somethingLike(Tuple2<Long, Long> t) {
        return ((double) Math.abs(t._1 - t._2) / Math.max(t._1, t._2)) < 0.3;
    }
}
