package ru.yandex.chemodan.ratelimiter.chunk;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;

/**
 * @author yashunsky
 */
public class ChunkRateLimiterWithMeter extends ChunkRateLimiterImpl {
    private final Duration meterInterval;
    private final RequestsBatchBuffer latestRequests;

    public ChunkRateLimiterWithMeter(
            Function0<Double> rateSupplier,
            Function0<Long> maxTimeSlotSupplier,
            Function0<Option<Long>> maxAwaitTimeSupplier,
            Function0<Integer> defaultChunkSizeSupplier,
            Duration meterInterval,
            Duration meterResolution)
    {
        super(rateSupplier, maxTimeSlotSupplier, maxAwaitTimeSupplier, defaultChunkSizeSupplier);
        this.meterInterval = meterInterval;
        this.latestRequests = new RequestsBatchBuffer(meterInterval, meterResolution);
    }

    public ChunkRateLimiterWithMeter(
            Function0<Double> rateSupplier,
            Function0<Long> maxTimeSlotSupplier,
            Function0<Option<Long>> maxAwaitTimeSupplier,
            Function0<Integer> defaultChunkSizeSupplier,
            Duration meterInterval)
    {
        this(rateSupplier, maxTimeSlotSupplier, maxAwaitTimeSupplier, defaultChunkSizeSupplier,
                meterInterval, Duration.standardSeconds(1));
    }


    public double getRealRate() {
        int totalSize = latestRequests.getCount();
        return ((double) totalSize) / meterInterval.getMillis();
    }

    public double getUsage() {
        double expectedRate = rateSupplier.apply();
        return expectedRate > 0 ? getRealRate() / expectedRate : -1;
    }

    @Override
    protected void onActionDone(int chunkSize) {
        latestRequests.add(chunkSize);
    }

    @Override
    public String toString() {
        return "limit: " + String.format("%.2f", rateSupplier.apply()) + " (" + (int) (getUsage() * 100) + "% used)";
    }
}
