package ru.yandex.chemodan.ratelimiter.chunk;

import java.util.concurrent.atomic.AtomicIntegerArray;
import java.util.concurrent.atomic.AtomicLong;

import org.joda.time.Duration;

/**
 * @author yashunsky
 */
public class RequestsBatchBuffer {
    private final AtomicIntegerArray requests;
    private final AtomicLong lastAdded;
    private final long batchDuration;
    private final int bufferSize;

    public RequestsBatchBuffer(Duration bufferDuration, Duration resolution) {
        this.bufferSize = Math.max(10, (int) (bufferDuration.getMillis() / resolution.getMillis()));
        this.batchDuration = bufferDuration.getMillis() / this.bufferSize;
        this.requests = new AtomicIntegerArray(bufferSize);
        this.lastAdded = new AtomicLong(0);
    }

    private long now() {
        return System.currentTimeMillis();
    }

    public void add(int count) {
        long to = now();
        long from = lastAdded.getAndSet(to);

        int toReset = (int) Math.min(((to - from) / batchDuration), (long) bufferSize + 1);
        int currentBatch = getBatchId(to);
        if (currentBatch != getBatchId(from)) {
            requests.set(currentBatch, 0);
        }

        for (int i=1; i < toReset; i++) {
            int position = (currentBatch - i + bufferSize) % bufferSize;
            requests.set(position, 0);
        }
        requests.addAndGet(currentBatch, count);
    }

    private int getBatchId(long timestamp) {
        return (int) ((timestamp / batchDuration) % bufferSize);
    }

    public int getCount() {
        add(0);
        int result = 0;
        for (int i=0; i < bufferSize; i++) {
            result += requests.get(i);
        }
        return result;
    }
}
