package ru.yandex.chemodan.ratelimiter.chunk.auto;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiterWithMeter;
import ru.yandex.chemodan.util.yasm.monitor.YasmHostStatus;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.commune.db.shard2.Shard2;
import ru.yandex.misc.db.masterSlave.DataSourceWithStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author yashunsky
 */
public class AutoRwRateLimiterImpl implements AutoRwRateLimiter {
    public static final double MIN_RATE = 0.001;

    private static final Logger logger = LoggerFactory.getLogger(AutoRwRateLimiterImpl.class);
    private final YasmMonitor yasmMonitor;
    private final MetricsConfiguration metricsConfiguration;
    private final RateLimitersMetrics rateLimitersMetrics;

    private final Function0<Double> getInitialRateF;
    private final Function<Integer, Double> getMaxRateF;
    private final Function0<Double> getRateIncStepF;
    private final Function0<Double> getRateDecStepF;
    private final Function0<Double> getReadRateCoeffF;
    private final Function0<Double> getWriteRateCoeffF;
    private final Function0<Double> getMinUsageF;
    private final Function0<ListF<String>> getIgnoredHostsF;

    private final Function0<Boolean> getIsAutoEnabledF;

    private final Duration averageInterval;

    private final Shard2 shard;
    private final MasterSlave masterSlave;

    private volatile double rate;
    private volatile boolean hasAvailableHosts;

    private final ChunkRateLimiterWithMeter readRateLimiter;
    private final ChunkRateLimiterWithMeter writeRateLimiter;

    public AutoRwRateLimiterImpl(YasmMonitor yasmMonitor,
            MetricsConfiguration metricsConfiguration,
            RateLimitersMetrics rateLimitersMetrics,
            Function0<Double> getInitialRateF,
            Function<Integer, Double> getMaxRateF,
            Function0<Double> getRateIncStepF,
            Function0<Double> getRateDecStepF,
            Function0<Double> getReadRateCoeffF,
            Function0<Double> getWriteRateCoeffF,
            Function0<Double> getMinUsageF,
            Function0<ListF<String>> getIgnoredHostsF,
            Function0<Boolean> getIsAutoEnabledF,
            Function0<Long> getMaxTimeSlotF,
            Function0<Option<Long>> getMaxAwaitTimeF,
            Function0<Integer> getDefaultChunkSizeF,
            Duration meterInterval,
            Duration averageInterval,
            Shard2 shard,
            MasterSlave masterSlave)
    {
        this.yasmMonitor = yasmMonitor;
        this.metricsConfiguration = metricsConfiguration;
        this.rateLimitersMetrics = rateLimitersMetrics;

        this.getInitialRateF = getInitialRateF;
        this.getMaxRateF = getMaxRateF;
        this.getRateIncStepF = getRateIncStepF;
        this.getRateDecStepF = getRateDecStepF;
        this.getReadRateCoeffF = getReadRateCoeffF;
        this.getWriteRateCoeffF = getWriteRateCoeffF;
        this.getMinUsageF = getMinUsageF;
        this.getIgnoredHostsF = getIgnoredHostsF;
        this.getIsAutoEnabledF = getIsAutoEnabledF;
        this.averageInterval = averageInterval;
        this.shard = shard;
        this.masterSlave = masterSlave;

        this.rate = getInitialRateF.apply();

        this.hasAvailableHosts = false;

        this.readRateLimiter = new ChunkRateLimiterWithMeter(
                this::getReadRate, getMaxTimeSlotF, getMaxAwaitTimeF, getDefaultChunkSizeF, meterInterval);

        this.writeRateLimiter = new ChunkRateLimiterWithMeter(
                this::getWriteRate, getMaxTimeSlotF, getMaxAwaitTimeF, getDefaultChunkSizeF, meterInterval);

    }

    public ChunkRateLimiterWithMeter getReadRateLimiter() {
        return readRateLimiter;
    }

    public ChunkRateLimiterWithMeter getWriteRateLimiter() {
        return writeRateLimiter;
    }

    @Override
    public boolean hasAwaitingRequests() {
        return readRateLimiter.hasAwaitingRequests() || writeRateLimiter.hasAwaitingRequests();
    }

    @Override
    public boolean hasAvailableHosts() {
        return hasAvailableHosts;
    }

    private double getReadRate() {
        return rate * getReadRateCoeffF.apply();
    }

    private double getWriteRate() {
        return rate * getWriteRateCoeffF.apply();
    }

    private ListF<DataSourceWithStatus> getDataSources() {
        return DataSourceHostsUtils.getDataSources(shard, "ds");
    }

    private Option<HostMasterSlave> getHostIfSuits(DataSourceWithStatus dsws) {
        Option<HostMasterSlave> hostMasterSlave =
                DataSourceHostsUtils.getHostMasterSlave(dsws, getIgnoredHostsF.apply());

        if (masterSlave.isMaster()) {
            return hostMasterSlave;
        } else {
            return hostMasterSlave.filter(hms -> hms.getMasterSlave() == masterSlave);
        }
    }

    private YasmHostStatus getHostStatus(HostMasterSlave hostMasterSlave) {
        String host = hostMasterSlave.getHost();
        MasterSlave masterSlave = hostMasterSlave.getMasterSlave();
        return yasmMonitor.getHostStatus(host, metricsConfiguration.getRanges(host, masterSlave), averageInterval);
    }

    private void approachRateTo(double estimatedRate) {
        double step;
        log("rate: " + rate + ", estimatedRate: " + estimatedRate);
        if (rate < estimatedRate) {
            step = getRateIncStepF.apply();
            log("inc by " + step);
            rate = limitedRate(Math.min(Math.max(rate, 0) + step, estimatedRate));
        } else if (rate > estimatedRate) {
            step = getRateDecStepF.apply();
            log("dec by " + step);
            rate = limitedRate(Math.max(rate - step, estimatedRate));
        }
        log("new rate: " + rate);
    }

    private double limitedRate(double rate) {
        return Math.min(rate, getMaxRateF.apply(shard.getShardInfo().getId()));
    }

    @Override
    public synchronized void maintain() {
        StatusAndUsage statusAndUsage = getStatusAndUpdateMetrics();

        if (!getIsAutoEnabledF.apply()) {
            rate = getInitialRateF.apply();
            return;
        }

        double minUsage = getMinUsageF.apply();
        YasmHostStatus status = statusAndUsage.shardStatus;
        double usage = statusAndUsage.limiterUsage;

        switch (status) {
            case OK:
                log("usage: " + usage + ", min usage: " + minUsage);
                if (usage > minUsage) {
                    approachRateTo(Double.MAX_VALUE);
                } else {
                    approachRateTo(getInitialRateF.apply());
                }
                break;
            case WARN:
                if (rate > 0) {
                    approachRateTo(MIN_RATE);
                }
                break;
            case CRIT:
                rate = -1;
                break;
        }
    }

    private StatusAndUsage getStatusAndUpdateMetrics() {
        Option<YasmHostStatus> statusO = getDataSources()
                .filterMap(this::getHostIfSuits).map(this::getHostStatus)
                .reduceRightO(YasmHostStatus::worst);
        log("shard status " + statusO);
        hasAvailableHosts = statusO.isPresent();

        YasmHostStatus status = statusO.getOrElse(YasmHostStatus.CRIT);
        double usage = (masterSlave.isMaster() ? writeRateLimiter : readRateLimiter).getUsage();
        double realRate = (masterSlave.isMaster() ? writeRateLimiter : readRateLimiter).getRealRate();
        double rateCoeff = (masterSlave.isMaster() ? getWriteRateCoeffF : getReadRateCoeffF).apply();
        rateLimitersMetrics.shardsHealth.set(status.getCritLevel(), getMetricName());
        rateLimitersMetrics.rates.set(Math.max(rate * rateCoeff, 0), getMetricName().withSuffix("target_rate"));
        rateLimitersMetrics.rates.set(Math.max(realRate, 0), getMetricName().withSuffix("real_rate"));

        return new StatusAndUsage(status, usage);
    }

    public void updateMetrics() {
        getStatusAndUpdateMetrics();
    }

    private String getId() {
        return "Shard " + shard.getShardInfo().getId() + "-" + String.valueOf(masterSlave).toLowerCase();
    }

    private MetricName getMetricName() {
        return new MetricName(getId().replace(" ", ""));
    }

    private void log(String message) {
        logger.info("{}: {}", getId(), message);
    }

    @Override
    public String toString() {
        return getId() + "\n" +
                "Hosts available: " + hasAvailableHosts + "\n" +
                "Read: " + readRateLimiter + "\n" +
                "Write: " + writeRateLimiter + "\n" +
                "Auto enabled: " + getIsAutoEnabledF.apply();
    }

    public double getRateForTest() {
        return rate;
    }

    @AllArgsConstructor
    @Data
    private static class StatusAndUsage {
        private final YasmHostStatus shardStatus;
        private final double limiterUsage;
    }
}
