package ru.yandex.chemodan.ratelimiter.chunk.auto;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.util.yasm.monitor.YasmHostStatus;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.commune.db.shard2.ShardManager2;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public abstract class AutoRwRateLimiterMapSupport extends AutoRwRateLimiterMap<HostKey, AutoRwRateLimiter> {
    private final ShardManager2 dataShardManager;
    private final YasmMonitor yasmMonitor;
    private final MetricsConfiguration metricsConfiguration;
    private final RateLimitersMetrics rateLimitersMetrics;

    private final Function0<Double> getInitialRateF;
    private final Function<Integer, Double> getMaxRateF;
    private final Function0<Double> getRateIncStepF;
    private final Function0<Double> getRateDecStepF;
    private final Function0<Double> getReadRateCoeffF;
    private final Function0<Double> getWriteRateCoeffF;
    private final Function0<Double> getMinUsageF;
    private final Function0<ListF<String>> getIgnoredHostsF;
    private final Function0<Boolean> getIsAutoEnabledF;
    private final Function0<Long> getMaxTimeSlotF;
    private final Function0<Option<Long>> getMaxAwaitTimeF;
    private final Function0<Integer> getDefaultChunkSizeF;
    private final Duration meterInterval;
    private final Duration averageInterval;

    @Override
    protected AutoRwRateLimiter limiterConstructor(HostKey key) {
        return new AutoRwRateLimiterImpl(
                yasmMonitor, metricsConfiguration, rateLimitersMetrics,
                getInitialRateF, getMaxRateF, getRateIncStepF, getRateDecStepF, getReadRateCoeffF,
                getWriteRateCoeffF, getMinUsageF, getIgnoredHostsF, getIsAutoEnabledF,
                getMaxTimeSlotF, getMaxAwaitTimeF, getDefaultChunkSizeF,
                meterInterval, averageInterval,
                dataShardManager.getShard(key.getShardId()), key.getMasterSlave());
    }

    public boolean isMigrationPossible(ListF<Integer> shardIds) {
        if (shardIds.isEmpty() || !getIsAutoEnabledF.apply()) {
            return true;
        }

        YasmHostStatus worstHostStatus = dataShardManager.shards()
                .filter(shard -> shardIds.containsTs(shard.getShardInfo().getId()))
                .flatMap(shard -> DataSourceHostsUtils.getDataSources(shard, "ds"))
                .filterMap(dsws -> DataSourceHostsUtils.getHostMasterSlave(dsws, getIgnoredHostsF.apply()))
                .map(hostMasterSlave -> yasmMonitor.getHostStatus(
                        hostMasterSlave.getHost(),
                        metricsConfiguration.getRanges(hostMasterSlave.getHost(), hostMasterSlave.getMasterSlave()),
                        averageInterval)
                ).reduceRightO(YasmHostStatus::worst).getOrElse(YasmHostStatus.CRIT);

        return worstHostStatus != YasmHostStatus.CRIT;
    }
}
