package ru.yandex.chemodan.ratelimiter.chunk.auto;

import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.joda.time.Duration;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.yasm.monitor.YasmHostStatus;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.commune.db.shard.ShardInfo;
import ru.yandex.commune.db.shard2.Shard2;
import ru.yandex.misc.db.ExposeUrlDataSource;
import ru.yandex.misc.db.masterSlave.DataSourceWithStatus;
import ru.yandex.misc.db.masterSlave.dynamic.DynamicMasterSlaveDataSource;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
public class AutoRwRateLimiterTest {
    private YasmMonitor yasmMonitor;
    private MetricsConfiguration metricsConfiguration;
    private RateLimitersMetrics innerMetrics;
    private AutoRwRateLimiterImpl limiter;

    private final double initialRate = 10;
    private final double rateStep = 1.5;
    private final double readRateCoeff = 1.0;
    private final double writeRateCoeff = 0.5;
    private final double minUsage = 0.7;
    private final long maxTimeSlot = 1000;
    private final long maxAwaitTime = 10000;
    private final int defaultChunkSize = 100;
    private final Duration meterInterval = Duration.standardSeconds(1);
    private final Duration averageInterval = Duration.standardSeconds(1);

    @Before
    public void setup() {
        yasmMonitor = Mockito.mock(YasmMonitor.class);
        metricsConfiguration = Mockito.mock(MetricsConfiguration.class);
        innerMetrics = new RateLimitersMetrics("test");

        ExposeUrlDataSource ds = Mockito.mock(ExposeUrlDataSource.class);
        Mockito.when(ds.url()).thenReturn(Optional.of("jdbc:postgresql://apidb01e.disk.yandex.net:6432/api_disk_data"));

        DataSourceWithStatus dsws = Mockito.mock(DataSourceWithStatus.class);
        Mockito.when(dsws.isAvailable()).thenReturn(Optional.of(true));
        Mockito.when(dsws.isMaster()).thenReturn(Optional.of(true));
        Mockito.when(dsws.getDs()).thenReturn(ds);

        DynamicMasterSlaveDataSource dataSource = Mockito.mock(DynamicMasterSlaveDataSource.class);
        Mockito.when(dataSource.getDataSourcesWithStatuses(Mockito.anyString())).thenReturn(Cf.list(dsws));

        ShardInfo shardInfo = Mockito.mock(ShardInfo.class);

        Shard2 shard = Mockito.mock(Shard2.class);
        Mockito.when(shard.getDataSource()).thenReturn(dataSource);
        Mockito.when(shard.getShardInfo()).thenReturn(shardInfo);

        limiter = new AutoRwRateLimiterImpl(yasmMonitor, metricsConfiguration, innerMetrics,
                () -> initialRate, shardId -> Double.MAX_VALUE, () -> rateStep, () -> rateStep, () -> readRateCoeff,
                () -> writeRateCoeff, () -> minUsage, Cf::list, () -> true,
                () -> maxTimeSlot, () -> Option.of(maxAwaitTime),
                () -> defaultChunkSize, meterInterval, averageInterval,
                shard, MasterSlave.MASTER);
    }

    @Test
    public void hostDegradationWithFurtherRestore() {
        Mockito.when(yasmMonitor.getHostStatus(Mockito.anyString(), Mockito.any(), Mockito.any())).thenReturn(
                YasmHostStatus.OK,
                YasmHostStatus.OK,
                YasmHostStatus.OK,
                YasmHostStatus.WARN,
                YasmHostStatus.CRIT,
                YasmHostStatus.WARN,
                YasmHostStatus.OK);

        initialCharge();

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), initialRate + rateStep, 0.1);

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), initialRate + 2 * rateStep, 0.1);

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), initialRate + rateStep, 0.1);

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), -1, 0.1);

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), -1, 0.1);

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), rateStep, 0.1);
    }

    @Test
    public void everythingDone() {
        Mockito.when(yasmMonitor.getHostStatus(Mockito.anyString(), Mockito.any(), Mockito.any()))
                .thenReturn(YasmHostStatus.OK);

        initialCharge();

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), initialRate + rateStep, 0.1);

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), initialRate + 2 * rateStep, 0.1);

        while (limiter.getWriteRateLimiter().hasAwaitingRequests()) {
            ThreadUtils.sleep(100);
        }

        double rateOnEverythingDone = limiter.getRateForTest();
        Assert.gt(rateOnEverythingDone, initialRate);

        ThreadUtils.sleep(2000);
        Assert.lt(limiter.getWriteRateLimiter().getUsage(), minUsage);

        limiter.maintain();
        Assert.equals(limiter.getRateForTest(), rateOnEverythingDone - rateStep, 0.1);

    }

    private void initialCharge() {
        limiter.maintain();
        charge(100);
        ThreadUtils.sleep(1000);
    }

    private void charge(int count) {
        ExecutorService service = Executors.newFixedThreadPool(count);
        for (int i = 0; i < count; i++) {
            service.submit(() -> limiter.getWriteRateLimiter().acquirePermitAndExecuteV(100, () -> {}));
        }
    }
}
