package ru.yandex.chemodan.ratelimiter.yarl;

import java.util.function.Supplier;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.ratelimiter.RateLimitExceededException;
import ru.yandex.chemodan.web.InterceptorOrders;
import ru.yandex.commune.a3.action.intercept.ActionInvocationInterceptor;
import ru.yandex.commune.a3.action.invoke.ActionInvocation;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author friendlyevil
 */
public abstract class YarlBaseInterceptor implements ActionInvocationInterceptor {
    private static final Logger logger = LoggerFactory.getLogger(YarlBaseInterceptor.class);

    public static final String RATE_LIMITER_IS_CHECKED = "X-RateLimiterChecked";
    private final YarlHttpClient yarlHttpClient;
    private final Supplier<Boolean> isDryRun;

    public YarlBaseInterceptor(YarlHttpClient yarlHttpClient, Supplier<Boolean> isDryRun) {
        this.yarlHttpClient = yarlHttpClient;
        this.isDryRun = isDryRun;
    }

    public Object intercept(ActionInvocation invocation) throws Exception {
        HttpServletRequestX request = invocation.getWebRequest().getHttpServletRequest();

        if (!needRateLimitCheck(request)) {
            return invocation.invoke();
        }

        Option<String> clientIdO = getClientId(request);
        if (clientIdO.isEmpty()) {
            logger.error("No request to ratelimiter: client id not found in attributes");
            return invocation.invoke();
        }

        int weight = getWeight(invocation);
        RateLimiterStatus rateLimiterStatus = yarlHttpClient.checkLimit(clientPrefix() + clientIdO.get(), weight);

        if (!isDryRun.get() && rateLimiterStatus == RateLimiterStatus.REJECTED) {
            throw new RateLimitExceededException();
        }

        afterSuccessCheck(request);
        return invocation.invoke();
    }

    protected abstract String clientPrefix();

    protected abstract Option<String> getClientId(HttpServletRequestX request);

    protected int getWeight(ActionInvocation invocation) {
        return 1;
    }

    protected boolean needRateLimitCheck(HttpServletRequestX request) {
        return request.getAttribute(RATE_LIMITER_IS_CHECKED) == null && request.getHeader(RATE_LIMITER_IS_CHECKED) == null;
    }

    protected void afterSuccessCheck(HttpServletRequestX request) {
        request.setAttribute(RATE_LIMITER_IS_CHECKED, "1");
    }

    @Override
    public int getOrder() {
        return InterceptorOrders.RATE_LIMITER_INTERCEPTOR_ORDER;
    }
}
