package ru.yandex.chemodan.scripter.admin;

import java.util.Set;

import lombok.Data;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.boot.admin.idm.IdmAdminRolesService;
import ru.yandex.chemodan.scripter.ScriptInvocationResult;
import ru.yandex.chemodan.scripter.ScripterManager;
import ru.yandex.chemodan.scripter.ScripterScript;
import ru.yandex.chemodan.util.auth.YateamAuthUtils;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
@ActionContainer
public class ScripterAdminPage {
    private final ScripterManager scripterManager;

    public ScripterAdminPage(ScripterManager scripterManager) {
        this.scripterManager = scripterManager;
    }

    @ZAction(defaultAction = true)
    @Path("/scripter")
    public ListScripts index(@SpecialParam HttpServletRequestX reqX) {
        return new ListScripts(
                scripterManager.getAll().map(s -> addRights(s, reqX)), isCanEdit(reqX));
    }

    @ZAction(file = "edit.xsl")
    @Path(value = "/scripter/edit")
    public ListScripts get(
            @RequestParam("id") String scriptId,
            @SpecialParam HttpServletRequestX reqX)
    {
        return new ListScripts(Cf.list(scripterManager.get(scriptId)).map(s -> addRights(s, reqX)), isCanEdit(reqX));
    }

    @Path(value = "/scripter/save_code", methods = HttpMethod.POST)
    public void get(
            @RequestParam("id")
            String scriptId,
            @RequestParam("description")
            String description,
            @RequestParam("jsCode")
            String jsCode)
    {
        jsCode = StringUtils.trimToEmpty(jsCode);
        if (jsCode.isEmpty()) {
            jsCode = "print('HelloWorld');";
        }

        ScripterScript script = scripterManager
                .getO(scriptId)
                .getOrElse(new ScripterScript(scriptId))
                .toBuilder()
                .description(description)
                .jsCode(jsCode)
                .build();

        scripterManager.save(script);

        throw new ZRedirectException("/z/scripter/edit?id=" + scriptId);
    }

    @Path(value = "/scripter/add_parameter", methods = HttpMethod.POST)
    public void addParameter(
            @RequestParam("id")
            String scriptId,
            @RequestParam("name")
            String name,
            @RequestParam("description")
            String description)
    {
        ScripterScript script = scripterManager.get(scriptId);

        script = script
                .toBuilder()
                .parameters(script.parameters.plus(new ScripterScript.Parameter(name, description)))
                .build();

        scripterManager.save(script);
        throw new ZRedirectException("/z/scripter/edit?id=" + scriptId);
    }

    @Path(value = "/scripter/delete", methods = HttpMethod.POST)
    public void delete(@RequestParam("id") String scriptId) {
        scripterManager.remove(scriptId);
        throw new ZRedirectException("/z/scripter");
    }

    @Path(value = "/scripter/remove_parameter", methods = HttpMethod.POST)
    public void removeParameter(
            @RequestParam("id")
            String scriptId,
            @RequestParam("name")
            String name)
    {
        ScripterScript script = scripterManager.get(scriptId);

        script = script
                .toBuilder()
                .parameters(script.parameters.filterNot(p -> p.name.equals(name)))
                .build();

        scripterManager.save(script);
        throw new ZRedirectException("/z/scripter/edit?id=" + scriptId);
    }

    @ZAction(file = "invoke.xsl")
    @Path(value = "/scripter/invoke", methods = HttpMethod.POST)
    public ScriptWithInvocation invoke(
            @RequestParam("id")
            String scriptId,
            @SpecialParam
            HttpServletRequestX reqX)
    {
        ScripterScript script = scripterManager.get(scriptId);

        MapF<String, String> parameters = script.parameters
                .map(p -> p.name)
                .toMapMappingToValue(name -> reqX.getParameterO(name).getOrElse(""));

        return new ScriptWithInvocation(
                addRights(script, reqX),
                scripterManager.invoke(scriptId, parameters)
        );
    }

    private ScriptWithRights addRights(ScripterScript script, HttpServletRequestX reqX) {
        return new ScriptWithRights(script, isCanInvoke(script, reqX), isCanEdit(reqX));
    }

    private boolean isCanInvoke(ScripterScript script, HttpServletRequestX reqX) {
        if (isNotProd()) {
            boolean denyInvoke = reqX.getParameterO("invoke").isSome("false");
            return !denyInvoke;
        }

        return script.invokers.containsTs(getLogin(reqX));
    }

    private boolean isCanEdit(HttpServletRequestX reqX) {
        if (isNotProd()) {
            boolean denyEdit = reqX.getParameterO("edit").isSome("false");
            return !denyEdit;
        }

        Set<String> roles = IdmAdminRolesService.getRoles(reqX);
        return roles.contains("EXECUTE");
    }

    private boolean isNotProd() {
        return EnvironmentType.getActive() != EnvironmentType.PRODUCTION && EnvironmentType.getActive() != EnvironmentType.PRESTABLE;
    }

    private String getLogin(HttpServletRequestX reqX) {
        return YateamAuthUtils.getLoginFromAttributeO(reqX)
                .orElse(Option.when(EnvironmentType.getActive() != EnvironmentType.PRODUCTION, "empty"))
                .getOrThrow("No login");
    }

    @Data
    @BenderBindAllFields
    @XmlRootElement(name = "content")
    private static class ListScripts {
        public final ListF<ScriptWithRights> scripts;
        public final boolean canCreate;
    }

    @Data
    @BenderBindAllFields
    @XmlRootElement(name = "content")
    private static class ScriptWithRights {
        @BenderFlatten
        public final ScripterScript script;

        public final boolean canInvoke;
        public final boolean canEdit;
    }

    @Data
    @BenderBindAllFields
    @XmlRootElement(name = "content")
    private static class ScriptWithInvocation {
        public final ScriptWithRights script;
        public final ScriptInvocationResult invocation;
    }
}
